<?php
session_start();

// Incluir sistema de control de acceso
require '../../backend/php/check_access.php';

// Verificar permisos
if(!hasModulePermission('cursos') && !hasSubmenuPermission('cursos_registro')){
    header('location: ../acceso_denegado.php');
    exit;
}

require '../../backend/bd/Conexion.php';
?>
<!DOCTYPE html>
<html lang="es">
<head>
    <meta charset="UTF-8">
    <link href='https://unpkg.com/boxicons@2.0.9/css/boxicons.min.css' rel='stylesheet'>
    <link rel="stylesheet" href="../../backend/css/admin.css?v=7.0">
    <link rel="icon" type="image/png" sizes="96x96" href="../../backend/img/ico.svg">
    <link rel="stylesheet" href="https://cdnjs.cloudflare.com/ajax/libs/limonte-sweetalert2/6.6.9/sweetalert2.min.css">
    <script src="https://cdn.jsdelivr.net/npm/apexcharts"></script>
    <title>EFAC | Gestión de Escuelas</title>
    
    <style>
        /* Custom Styles for 4-Column Layout */
        .hierarchical-container {
            display: grid;
            grid-template-columns: repeat(3, minmax(280px, 1fr));
            gap: 20px;
            margin-top: 20px;
        }

        .column-card {
            background: white;
            border-radius: 10px;
            box-shadow: 0 4px 6px rgba(0,0,0,0.1);
            display: flex;
            flex-direction: column;
            height: 600px; /* Fixed height for scrollable lists */
            border: 1px solid #e0e0e0;
        }

        .column-header {
            padding: 20px;
            border-bottom: 2px solid #f0f0f0;
            display: flex;
            justify-content: space-between;
            align-items: center;
        }

        .column-header h2 {
            font-size: 1.2rem;
            color: #333;
            margin: 0;
            display: flex;
            align-items: center;
            gap: 10px;
        }

        .column-header .count-badge {
            background: #e0e7ff;
            color: #4338ca;
            padding: 2px 8px;
            border-radius: 12px;
            font-size: 0.8rem;
            font-weight: bold;
        }

        .column-body {
            flex: 1;
            overflow-y: auto;
            padding: 10px;
            background: #f9fafb;
        }

        .item-list {
            list-style: none;
            padding: 0;
            margin: 0;
        }

        .list-item {
            background: white;
            padding: 15px;
            border-radius: 8px;
            margin-bottom: 10px;
            cursor: pointer;
            border: 1px solid #e0e0e0;
            transition: all 0.2s ease;
            position: relative;
        }

        .list-item:hover {
            transform: translateY(-2px);
            box-shadow: 0 4px 12px rgba(0,0,0,0.05);
            border-color: #6366f1;
        }

        .list-item.active {
            background: #e0e7ff;
            border-color: #4338ca;
        }
        
        .list-item:hover .edit-btn {
            display: inline-flex;
        }

        .edit-btn {
            background: #e5e7eb;
            color: #4b5563;
            border: none;
            border-radius: 4px;
            padding: 4px 8px;
            font-size: 1.1rem;
            cursor: pointer;
            position: absolute;
            right: 15px; /* Adjust as needed */
            top: 15px;
            display: none; /* Hidden by default */
            align-items: center;
            justify-content: center;
            transition: background 0.2s;
            z-index: 5;
        }

        .edit-btn:hover {
            background: #dbeafe;
            color: #2563eb;
        }

        .list-item-title {
            font-weight: bold;
            color: #1f2937;
            display: block;
            margin-bottom: 4px;
        }

        .list-item-subtitle {
            font-size: 0.85rem;
            color: #6b7280;
        }

        .add-btn {
            background: #f59e0b; /* Orange/Yellow from example */
            color: white;
            border: none;
            width: 36px;
            height: 36px;
            border-radius: 50%;
            display: flex;
            align-items: center;
            justify-content: center;
            cursor: pointer;
            transition: background 0.2s;
            font-size: 1.2rem;
        }

        .add-btn:hover {
            background: #d97706;
        }

        .disabled-link {
            pointer-events: none;
            opacity: 0.5;
        }
        
        .group-card-selectable.active {
            border-color: #4338ca !important;
            box-shadow: 0 0 0 2px rgba(67,56,202,0.15);
        }
        .group-download-btn {
            border: none;
            background: #dbeafe;
            color: #1d4ed8;
            border-radius: 4px;
            padding: 4px 6px;
            margin-right: 6px;
            cursor: pointer;
        }
        .group-download-btn:hover {
            background: #bfdbfe;
        }
        
        .disabled-overlay {
            position: absolute;
            top: 0;
            left: 0;
            right: 0;
            bottom: 0;
            background: rgba(255,255,255,0.7);
            z-index: 10;
            display: flex;
            align-items: center;
            justify-content: center;
            border-radius: 10px;
            color: #666;
            font-weight: bold;
        }

        /* Modal Styles */
        #gestion-grupos-container {
            position: relative;
            z-index: 100;
            width: 100%;
            /* display: none;  REMOVED TO FORCE VISIBILITY DEBUG */
        }
        
        .modal {
            display: none; 
            position: fixed; 
            z-index: 1000; 
            left: 0;
            top: 0;
            width: 100%; 
            height: 100%; 
            overflow: auto; 
            background-color: rgba(0,0,0,0.5); 
            backdrop-filter: blur(4px);
        }

        .modal-content {
            background-color: #fefefe;
            margin: 10% auto; 
            padding: 0;
            border: 1px solid #888;
            width: 100%;
            max-width: 500px;
            border-radius: 12px;
            box-shadow: 0 10px 25px rgba(0,0,0,0.2);
            animation: modalFadeIn 0.3s;
        }
        
        @keyframes modalFadeIn {
            from {opacity: 0; transform: translateY(-50px);}
            to {opacity: 1; transform: translateY(0);}
        }

        .modal-header {
            padding: 20px;
            background: #f8fafc;
            border-bottom: 1px solid #e2e8f0;
            border-radius: 12px 12px 0 0;
            display: flex;
            justify-content: space-between;
            align-items: center;
        }

        .modal-header h3 {
            margin: 0;
            color: #1e293b;
        }

        .close {
            color: #aaa;
            font-size: 28px;
            font-weight: bold;
            cursor: pointer;
        }

        .close:hover {
            color: #000;
        }

        .modal-body {
            padding: 20px;
        }

        .form-group {
            margin-bottom: 15px;
        }

        .form-group label {
            display: block;
            margin-bottom: 5px;
            font-weight: 500;
            color: #475569;
        }

        .form-group input, .form-group textarea, .form-group select {
            width: 100%;
            padding: 10px;
            border: 1px solid #cbd5e1;
            border-radius: 6px;
            font-size: 14px;
        }
        
        .form-group input:focus {
            outline: none;
            border-color: #6366f1;
            box-shadow: 0 0 0 3px rgba(99, 102, 241, 0.1);
        }

        .submit-btn {
            background: #4f46e5;
            color: white;
            padding: 12px;
            border: none;
            border-radius: 6px;
            width: 100%;
            font-weight: bold;
            cursor: pointer;
            transition: background 0.2s;
        }

        .submit-btn:hover {
            background: #4338ca;
        }

        .badge-info {
            background: #dbeafe;
            color: #1e40af;
            padding: 2px 6px;
            border-radius: 4px;
            font-size: 0.7rem;
            margin-left: 5px;
        }
    </style>
</head>
<body>
    
    <!-- SIDEBAR -->
    <?php include '../../backend/php/sidebar_menu.php'; ?>
    <!-- SIDEBAR -->

    <!-- NAVBAR -->
    <section id="content">
        <nav>
            <i class='bx bx-menu toggle-sidebar' ></i>
            <form action="#">
                <div class="form-group-search"> <!-- Styled separately if needed -->
                    <input type="text" placeholder="Search...">
                    <i class='bx bx-search icon' ></i>
                </div>
            </form>
            <span class="divider"></span>
            <div class="profile">
                <img src="../../backend/img/user-default.png" alt="">
                <ul class="profile-link">
                    <li><a href="../profile/mostrar.php"><i class='bx bxs-user-circle icon' ></i> Perfil</a></li>
                    <li><a href="../salir.php"><i class='bx bxs-log-out-circle' ></i> Salir</a></li>
                </ul>
            </div>
        </nav>

        <main>
            <h1 class="title">Gestión de Escuelas de Formación</h1>
            <ul class="breadcrumbs">
                <li><a href="../admin/escritorio.php">Home</a></li>
                <li class="divider">></li>
                <li><a href="#" class="active">Escuelas, Áreas y Cursos</a></li>
            </ul>

            <div class="hierarchical-container">
                
                <!-- CONTAINER 1: ESCUELAS -->
                <div class="column-card" id="col-escuelas">
                    <div class="column-header">
                        <h2>
                            <i class='bx bxs-institution'></i> Escuelas
                            <span class="count-badge" id="count-escuelas">0</span>
                        </h2>
                        <button class="add-btn" onclick="openModal('modalEscuela')"><i class='bx bx-plus'></i></button>
                    </div>
                    <div class="column-body">
                        <ul class="item-list" id="list-escuelas">
                            <!-- Items Loaded via AJAX -->
                            <li class="list-item" style="text-align: center; color: #999;">Cargando...</li>
                        </ul>
                    </div>
                </div>

                <!-- CONTAINER 2: ÁREAS -->
                <div class="column-card" id="col-areas" style="position: relative;">
                    <div class="disabled-overlay" id="overlay-areas">
                        Selecciona una Escuela
                    </div>
                    <div class="column-header">
                        <h2>
                            <i class='bx bxs-widget'></i> Áreas
                            <span class="count-badge" id="count-areas">0</span>
                        </h2>
                        <button class="add-btn" onclick="openModal('modalArea')"><i class='bx bx-plus'></i></button>
                    </div>
                    <div class="column-body">
                        <ul class="item-list" id="list-areas">
                            <!-- Items Loaded via AJAX -->
                        </ul>
                    </div>
                </div>

                <!-- CONTAINER 3: CURSOS -->
                <div class="column-card" id="col-cursos" style="position: relative;">
                    <div class="disabled-overlay" id="overlay-cursos">
                        Selecciona un Área
                    </div>
                    <div class="column-header">
                        <h2>
                            <i class='bx bxs-book-content'></i> Cursos
                            <span class="count-badge" id="count-cursos">0</span>
                        </h2>
                        <button class="add-btn" onclick="openModal('modalCurso')"><i class='bx bx-plus'></i></button>
                    </div>
                    <div class="column-body">
                        <ul class="item-list" id="list-cursos">
                            <!-- Items Loaded via AJAX -->
                        </ul>
                    </div>
                </div>
            </div>

            <!-- GROUP MANAGEMENT SECTION (Always block, we'll hide content if needed or just show empty) -->
            <div id="gestion-grupos-container" style="display:block; margin-top: 20px;">
                <div style="background: white; border-radius: 10px; box-shadow: 0 4px 6px rgba(0,0,0,0.1); padding: 20px; border: 1px solid #e0e0e0;">
                    
                    <div style="display:flex; justify-content:space-between; align-items:center; margin-bottom:20px; border-bottom: 2px solid #f0f0f0; padding-bottom:10px; gap:12px; flex-wrap:wrap;">
                        <div style="display:flex; flex-direction:column; gap:4px;">
                            <h2 style="font-size: 1.3rem; color: #333; margin: 0;">
                                <i class='bx bx-grid-alt'></i> Gestión de Grupos: <span id="gestion-curso-nombre" style="color:#4338ca;"></span>
                            </h2>
                            <div id="gestion-curso-meta" style="font-size:0.9rem; color:#666;"></div>
                        </div>
                        <div style="display:flex; gap:10px;">
                            <a id="btnDescargarCurso" href="#" target="_blank" rel="noopener" class="disabled-link" style="border:1px solid #4338ca; color:#4338ca; border-radius:999px; padding:8px 16px; text-decoration:none; display:flex; align-items:center; gap:6px;">
                                <i class='bx bx-download'></i> Descargar lista de curso
                            </a>
                            <a id="btnDescargarGrupos" href="#" target="_blank" rel="noopener" class="disabled-link" style="border:1px solid #0f766e; color:#0f766e; border-radius:999px; padding:8px 16px; text-decoration:none; display:flex; align-items:center; gap:6px;">
                                <i class='bx bx-file'></i> Descargar reporte grupos
                            </a>
                            <button class="add-btn" onclick="openModal('modalGrupo')"><i class='bx bx-plus'></i> Nuevo Grupo</button>
                        </div>
                    </div>

                    <div style="display: flex; gap: 20px; height: 600px;">
                        
                        <!-- LEFT PANEL: STUDENTS NO GROUP -->
                        <div style="width: 30%; display: flex; flex-direction: column; background: #f9fafb; border: 1px solid #e0e0e0; border-radius: 8px; padding: 10px;">
                            <h3 style="font-size: 1rem; color: #555; margin-bottom: 10px; text-align: center;">
                                Estudiantes Sin Grupo (<span id="count-sin-grupo">0</span>)
                            </h3>
                            <div id="lista-sin-grupo" class="droppable-zone" data-grupo-id="0" style="flex: 1; overflow-y: auto; padding: 5px;">
                                <!-- Draggable Items -->
                            </div>
                        </div>

                        <!-- RIGHT PANEL: GROUPS -->
                        <div style="width: 70%; display: flex; flex-direction: column; background: #fff; border: 1px solid #e0e0e0; border-radius: 8px; padding: 10px;">
                            <h3 style="font-size: 1rem; color: #555; margin-bottom: 10px;">Grupos Existentes</h3>
                            
                            <div id="grid-grupos" style="flex: 1; overflow-y: auto; display: grid; grid-template-columns: repeat(auto-fill, minmax(250px, 1fr)); grid-auto-rows: 300px; gap: 15px; padding: 5px;">
                                <!-- Group Cards (Droppable) -->
                            </div>
                        </div>

                    </div>
                </div>
            </div>
        </main>
    </section>

    <!-- MODAL ESCUELA -->
    <div id="modalEscuela" class="modal">
        <div class="modal-content">
            <div class="modal-header">
                <h3 id="modalTitleEscuela">Nueva Escuela de Formación</h3>
                <span class="close" onclick="closeModal('modalEscuela')">&times;</span>
            </div>
            <div class="modal-body">
                <form id="formEscuela">
                    <input type="hidden" name="action" id="escuelaAction" value="add_escuela">
                    <input type="hidden" name="id" id="escuelaId">
                    <!-- Código automático -->
                    <div class="form-group">
                        <label>Nombre</label>
                        <input type="text" name="nombre" id="escuelaNombre" required placeholder="Ej: Escuela de Artes">
                    </div>
                    <div class="form-group">
                        <label>Descripción</label>
                        <textarea name="descripcion" id="escuelaDescripcion" rows="3" placeholder="Breve descripción..."></textarea>
                    </div>
                    <button type="submit" class="submit-btn" id="btnSubmitEscuela">Guardar Escuela</button>
                </form>
            </div>
        </div>
    </div>

    <!-- MODAL ÁREA -->
    <div id="modalArea" class="modal">
        <div class="modal-content">
            <div class="modal-header">
                <h3 id="modalTitleArea">Nueva Área</h3>
                <span class="close" onclick="closeModal('modalArea')">&times;</span>
            </div>
            <div class="modal-body">
                <form id="formArea">
                    <input type="hidden" name="action" id="areaAction" value="add_area">
                    <input type="hidden" name="id" id="areaId">
                    <input type="hidden" name="id_escuela" id="input_area_escuela_id">
                    <div class="form-group">
                        <label>Escuela Padre</label>
                        <input type="text" id="display_escuela_name" disabled style="background:#f0f0f0;">
                    </div>
                    <!-- Código automático -->
                    <div class="form-group">
                        <label>Nombre</label>
                        <input type="text" name="nombre" id="areaNombre" required placeholder="Ej: Música">
                    </div>
                    <button type="submit" class="submit-btn" id="btnSubmitArea">Guardar Área</button>
                </form>
            </div>
        </div>
    </div>

    <!-- MODAL CURSO -->
    <div id="modalCurso" class="modal">
        <div class="modal-content">
            <div class="modal-header">
                <h3 id="modalTitleCurso">Nuevo Curso</h3>
                <span class="close" onclick="closeModal('modalCurso')">&times;</span>
            </div>
            <div class="modal-body">
                <form id="formCurso">
                    <input type="hidden" name="action" id="cursoAction" value="add_curso">
                    <input type="hidden" name="id" id="cursoId">
                    <input type="hidden" name="id_area" id="input_curso_area_id">
                    <div class="form-group">
                        <label>Área Padre</label>
                        <input type="text" id="display_area_name" disabled style="background:#f0f0f0;">
                    </div>
                    <div class="form-group">
                        <label>Nombre del Curso</label>
                        <input type="text" name="nombre" id="cursoNombre" required placeholder="Ej: Guitarra Básica">
                    </div>
                    <div class="form-group">
                        <label>Profesor / Instructor</label>
                        <select name="quien_lo_da" id="cursoProfesor" required>
                            <option value="">Seleccione un profesional...</option>
                            <!-- Populated via AJAX -->
                        </select>
                    </div>
                    <div class="form-group">
                        <label>Cupos Disponibles</label>
                        <input type="number" name="cupos" id="cursoCupos" required min="1" value="20" placeholder="Ej: 20">
                    </div>
                    <div class="form-group">
                        <label>¿Es titulado?</label>
                        <select name="es_titulado" id="cursoTitulado" required>
                            <option value="no">No</option>
                            <option value="si">Sí</option>
                        </select>
                    </div>
                    <div class="form-group">
                        <label>Descripción / Enfoque</label>
                        <input type="text" name="en_que" id="cursoDescripcion" placeholder="Opcional">
                    </div>
                    <button type="submit" class="submit-btn" id="btnSubmitCurso">Guardar Curso</button>
                </form>
            </div>
        </div>
    </div>

    <!-- MODAL GRUPO -->
    <div id="modalGrupo" class="modal">
        <div class="modal-content">
            <div class="modal-header">
                <h3 id="modalTitleGrupo">Nuevo Grupo</h3>
                <span class="close" onclick="closeModal('modalGrupo')">&times;</span>
            </div>
            <div class="modal-body">
                <form id="formGrupo">
                    <input type="hidden" name="action" id="grupoAction" value="add_grupo">
                    <input type="hidden" name="id" id="grupoId">
                    <input type="hidden" name="id_curso" id="grupoCursoId">
                    <div class="form-group">
                        <label>Nombre del Grupo</label>
                        <input type="text" name="nombre" id="grupoNombre" required placeholder="Ej: Grupo A - Mañanas">
                    </div>
                    
                    <div class="form-group">
                        <label>Días de Clase</label>
                        <div style="display: flex; gap: 10px; flex-wrap: wrap;">
                            <label><input type="checkbox" name="dias_grupo[]" value="Lunes"> Lun</label>
                            <label><input type="checkbox" name="dias_grupo[]" value="Martes"> Mar</label>
                            <label><input type="checkbox" name="dias_grupo[]" value="Miércoles"> Mié</label>
                            <label><input type="checkbox" name="dias_grupo[]" value="Jueves"> Jue</label>
                            <label><input type="checkbox" name="dias_grupo[]" value="Viernes"> Vie</label>
                            <label><input type="checkbox" name="dias_grupo[]" value="Sábado"> Sáb</label>
                            <label><input type="checkbox" name="dias_grupo[]" value="Domingo"> Dom</label>
                        </div>
                    </div>
                    
                    <div class="form-group">
                        <label>Horario de Clase</label>
                        <div style="display: flex; gap: 10px;">
                            <input type="time" id="horaInicioGrupo" style="width: 48%;">
                            <span style="align-self: center;">a</span>
                            <input type="time" id="horaFinGrupo" style="width: 48%;">
                        </div>
                        <input type="hidden" name="horario" id="grupoHorario">
                    </div>
                    
                    <!-- Quota Information -->
                    <div id="quotaInfo" style="background: #e0f2fe; border-left: 4px solid #0284c7; padding: 12px; margin-bottom: 15px; border-radius: 4px; display: none;">
                        <div style="font-weight: 600; color: #0c4a6e; margin-bottom: 5px;">📊 Distribución de Cupos:</div>
                        <div style="font-size: 0.9rem; color: #075985;">
                            <div>• Cupos del curso: <strong id="quotaCursoTotal">0</strong></div>
                            <div>• Ya asignados en grupos: <strong id="quotaAsignada">0</strong></div>
                            <div>• <span style="color: #15803d;">Disponibles: <strong id="quotaDisponible">0</strong></span></div>
                        </div>
                    </div>
                    
                    <div class="form-group">
                        <label>Cupos de este Grupo</label>
                        <input type="number" name="cupos" id="grupoCupos" required min="1" placeholder="Número de estudiantes">
                        <small style="color: #666; font-size: 0.85rem;">Los cupos se descuentan del total del curso</small>
                    </div>
                    <button type="submit" class="submit-btn" id="btnSubmitGrupo">Guardar Grupo</button>
                </form>
            </div>
        </div>
    </div>

    <script src="https://code.jquery.com/jquery-3.6.0.min.js"></script>
    <script src="https://cdnjs.cloudflare.com/ajax/libs/limonte-sweetalert2/6.6.9/sweetalert2.min.js"></script>
    <script src="../../backend/js/script.js"></script>

    <script>
        // State Management
        let currentEscuela = null;
        let currentArea = null;
        let currentCurso = null;

        $(document).ready(function() {
            // Initial Load
            loadEscuelas();
            loadProfesionales(); // Load instructors

            // Form Submissions
            $('#formEscuela').on('submit', function(e) {
                e.preventDefault();
                submitForm($(this), loadEscuelas, 'modalEscuela');
            });

            $(document).on('click', '.list-item', function(e) {
                // Only handle clicks for COURSES via this delegation
                if ($(this).parent().attr('id') !== 'list-cursos') return;

                // Prevent triggering if clicking edit button
                if ($(e.target).closest('.edit-btn').length) return;
                
                try {
                    const id = $(this).data('id');
                    const nombre = $(this).data('nombre');
                    const cupos = $(this).data('cupos');
                    
                    if(!id) {
                         alert("Error: Elemento sin ID. Recarga la página.");
                         return;
                    }

                    selectCursoNEW(this, id, nombre, cupos);
                } catch(err) {
                    console.error("ERROR EN CLICK HANDLER:", err);
                    alert("Error: " + err.message);
                }
            });

            // Event Delegation for Edit Course Button
            $('#list-cursos').on('click', '.edit-btn', function(e) {
                e.stopPropagation();
                const json = $(this).data('json');
                // Auto-parse if jQuery didn't already (it usually does for objects)
                const item = typeof json === 'string' ? JSON.parse(json) : json; 
                openEditCurso(e, item);
            });

            $('#formArea').on('submit', function(e) {
                e.preventDefault();
                submitForm($(this), function() { if(currentEscuela) loadAreas(currentEscuela.id); }, 'modalArea');
            });

            $('#formCurso').on('submit', function(e) {
                e.preventDefault();
                
                // Combine Time Inputs before Submit
                const start = $('#horaInicio').val();
                const end = $('#horaFin').val();
                
                let horarioStr = '';
                if(start && end) {
                    horarioStr = formatTime12h(start) + ' - ' + formatTime12h(end);
                } else if (start) {
                    horarioStr = formatTime12h(start);
                }
                
                $('#cursoHorario').val(horarioStr);

                submitForm($(this), function() { if(currentArea) loadCursos(currentArea.id); }, 'modalCurso');
            });

            $('#formGrupo').on('submit', function(e) {
                e.preventDefault();
                
                // Convert time range to formatted string
                const start = $('#horaInicioGrupo').val();
                const end = $('#horaFinGrupo').val();
                
                let horarioStr = '';
                if(start && end) {
                    horarioStr = formatTime12h(start) + ' - ' + formatTime12h(end);
                } else if (start) {
                    horarioStr = formatTime12h(start);
                }
                
                $('#grupoHorario').val(horarioStr);
                
                submitForm($(this), function() { if(currentCurso) loadGrupos(currentCurso.id); }, 'modalGrupo');
            });
        });

        // Helper to convert 24h "14:00" to "02:00 PM"
        function formatTime12h(time24) {
            if(!time24) return '';
            let [hours, minutes] = time24.split(':');
            let h = parseInt(hours);
            let ampm = h >= 12 ? 'PM' : 'AM';
            h = h % 12;
            h = h ? h : 12; // 0 should be 12
            return (h < 10 ? '0' + h : h) + ':' + minutes + ' ' + ampm;
        }

        // Helper to convert "02:00 PM" back to "14:00" for input type="time"
        function convertTo24h(time12h) {
            if(!time12h) return '';
            const [time, modifier] = time12h.split(' ');
            let [hours, minutes] = time.split(':');
            
            if (hours === '12') {
                hours = '00';
            }
            
            if (modifier === 'PM') {
                hours = parseInt(hours, 10) + 12;
            }
            
            return `${hours}:${minutes}`;
        }

        // --- LOADERS ---
        function loadProfesionales() {
            $.post('ajax_registro.php', { action: 'get_profesionales' }, function(response) {
                if (response.success) {
                    const select = $('#cursoProfesor');
                    select.find('option:not(:first)').remove(); // Keep default option
                    
                    response.data.forEach(item => {
                        const fullName = `${item.nodoc} ${item.apdoc}`;
                        select.append(`<option value="${fullName}">${fullName}</option>`);
                    });
                }
            }, 'json');
        }

        function loadEscuelas() {
            $.post('ajax_registro.php', { action: 'get_escuelas' }, function(response) {
                if (response.success) {
                    const list = $('#list-escuelas');
                    list.empty();
                    $('#count-escuelas').text(response.data.length);
                    
                    response.data.forEach(item => {
                        const li = $(`
                            <li class="list-item" onclick="if(!event.target.closest('.edit-btn')) selectEscuela(this, ${item.id}, '${item.nombre}')">
                                <span class="list-item-title">${item.nombre}</span>
                                <span class="list-item-subtitle">${item.codigo}</span>
                                <button class="edit-btn" onclick="openEditEscuela(event, ${item.id}, '${item.nombre}', '${item.descripcion || ''}')"><i class='bx bx-pencil'></i></button>
                            </li>
                        `);
                        list.append(li);
                    });
                }
            }, 'json');
        }

        function loadAreas(escuelaId) {
            $.post('ajax_registro.php', { action: 'get_areas', id_escuela: escuelaId }, function(response) {
                if (response.success) {
                    const list = $('#list-areas');
                    list.empty();
                    $('#count-areas').text(response.data.length);

                    if (response.data.length === 0) {
                        list.append('<li style="text-align:center;color:#ccc;padding:20px;">No hay áreas registradas</li>');
                    }

                    response.data.forEach(item => {
                        const li = $(`
                            <li class="list-item" onclick="if(!event.target.closest('.edit-btn')) selectArea(this, ${item.id}, '${item.nombre}')">
                                <span class="list-item-title">${item.nombre}</span>
                                <span class="list-item-subtitle">${item.codigo}</span>
                                <button class="edit-btn" onclick="openEditArea(event, ${item.id}, '${item.nombre}')"><i class='bx bx-pencil'></i></button>
                            </li>
                        `);
                        list.append(li);
                    });
                }
            }, 'json');
        }

        function loadCursos(areaId) {
            $.post('ajax_registro.php', { action: 'get_cursos', id_area: areaId }, function(response) {
                if (response.success) {
                    const list = $('#list-cursos');
                    list.empty();
                    $('#count-cursos').text(response.data.length);

                    if (response.data.length === 0) {
                        list.append('<li style="text-align:center;color:#ccc;padding:20px;">No hay cursos registrados</li>');
                    }

                    response.data.forEach(item => {
                        const dias = item.dias_clase || '';
                        const horario = item.horario_clase || '';
                        // Safe data attributes
                        const safeName = item.nombre.replace(/"/g, '&quot;');
                        
                        const li = $(`
                            <li class="list-item" 
                                data-id="${item.idcurso}" 
                                data-nombre="${safeName}" 
                                data-cupos="${item.cupos || 0}">
                                
                                <span class="list-item-title">${item.nombre}</span>
                                <span class="list-item-subtitle">Prof: ${item.quien_lo_da} | Cupos: ${item.cupos || 0}</span>
                                <div style="font-size: 0.8rem; color: #666; margin-top:2px;">
                                    ${dias ? '<i class="bx bx-calendar"></i> ' + dias : ''} 
                                    ${horario ? '<i class="bx bx-time"></i> ' + horario : ''}
                                </div>
                                <span class="badge-info">${item.es_titulado === 'si' ? 'Titulado' : 'Libre'}</span>
                                <button class="edit-btn" data-id="${item.idcurso}"><i class='bx bx-pencil'></i></button>
                            </li>
                        `);
                        list.append(li);
                    });
                }
            }, 'json');
        }
        
        function loadGrupos(cursoId) {
            $.post('ajax_registro.php', { action: 'get_grupos', id_curso: cursoId }, function(response) {
         if (response.success) {
             const list = $('#list-grupos');
             list.empty();
             $('#count-grupos').text(response.data.length);

             if (response.data.length === 0) {
                 list.append('<li style="text-align:center;color:#ccc;padding:10px;">No hay grupos creados</li>');
             }

             response.data.forEach(item => {
                 const horario = item.horario ? `<div style="font-size:0.8rem;color:#666"><i class="bx bx-time"></i> ${item.horario}</div>` : '';
                 const cupos = item.cupos > 0 ? `<span class="badge-info">Cupos: ${item.cupos}</span>` : '';
                 
                 const li = $(`
                     <li class="list-item" style="padding:10px;">
                         <span class="list-item-title" style="font-size:0.95rem;">${item.nombre}</span>
                         ${horario}
                         ${cupos}
                         <button class="edit-btn" onclick="deleteGrupo(${item.id})"><i class='bx bx-trash'></i></button>
                     </li>
                 `);
                 list.append(li);
             });
         }
     }, 'json');
 }

 function deleteGrupo(id) {
     swal({
         title: '¿Estás seguro?',
         text: "No podrás revertir esto!",
         type: 'warning',
         showCancelButton: true,
         confirmButtonColor: '#3085d6',
         cancelButtonColor: '#d33',
         confirmButtonText: 'Sí, eliminar!'
     }).then(function () {
         $.post('ajax_registro.php', { action: 'delete_grupo', id: id }, function(response) {
             if (response.success) {
                 swal('Eliminado!', 'El grupo ha sido eliminado.', 'success');
                 if(currentCurso) loadGrupos(currentCurso.id);
             } else {
                 swal('Error', response.message, 'error');
             }
         }, 'json');
     })
 }



        // --- SELECTION HANDLERS ---
        function selectEscuela(element, id, nombre) {
            // UI Update
            $('#list-escuelas .list-item').removeClass('active');
            $(element).addClass('active');

            // Logic
            currentEscuela = { id, nombre };
            currentArea = null; // Reset child
            
            // Enable Next Column
            $('#overlay-areas').hide();
            $('#input_area_escuela_id').val(id);
            $('#display_escuela_name').val(nombre);
            
            // Load Child Data
            loadAreas(id);

            // Reset Grandchild
            $('#overlay-cursos').css('display', 'flex'); // Show overlay
            $('#list-cursos').empty();
            $('#count-cursos').text('0');
            
            // Reset Report Module - wait for course selection
            $('#overlay-reporte').css('display', 'flex');
            $('#reporte-info').hide();
            $('#list-inscritos').empty();
            $('#count-inscritos').text('0');
        }

        function selectArea(element, id, nombre) {
            // UI Update
            $('#list-areas .list-item').removeClass('active');
            $(element).addClass('active');

            // Logic
            currentArea = { id, nombre };

            // Enable Next Column
            $('#overlay-cursos').hide();
            $('#input_curso_area_id').val(id);
            $('#display_area_name').val(nombre);

            // Load Child Data
            loadCursos(id);
            
            // Reset Report Module - wait for course selection
            $('#overlay-reporte').css('display', 'flex');
            $('#reporte-info').hide();
            $('#list-inscritos').empty();
            $('#count-inscritos').text('0');
        }

        function selectCursoNEW(element, id, nombre, cupos) {
            // UI Update
            $('#list-cursos .list-item').removeClass('active');
            $(element).addClass('active');

            // Logic
            currentCurso = { id, nombre, cupos };
            const baseUrl = 'reporte_inscritos_pdf.php';
            const cursoUrl = `${baseUrl}?scope=curso&id=${id}`;
            $('#btnDescargarCurso').attr('href', cursoUrl).removeClass('disabled-link');
            $('#gestion-curso-meta').text(`Cupos declarados: ${cupos || 'N/D'}`);
            $('#btnDescargarGrupos').attr('href', `${baseUrl}?scope=curso&id=${id}`).removeClass('disabled-link');

            // Show Bottom Section
            const container = $('#gestion-grupos-container');
            container.show();
            
            // Scroll to view
            $('html, body').animate({
                scrollTop: container.offset().top - 20
            }, 500);
            
            $('#gestion-curso-nombre').text(nombre);

            // Load Data
            loadCursoData(id);
        }

        function loadCursoData(cursoId) {
            // Show loading indicator
            $('#lista-sin-grupo').html('<div style="padding:10px;text-align:center;">Cargando...</div>');
            
            $.post('ajax_registro.php', { action: 'get_grupo_assignments', id_curso: cursoId }, function(response) {
                if (response.success) {
                    renderGroupsInterface(response.data);
                } else {
                    console.error("Server Error:", response.message);
                    swal('Error', 'Error al cargar datos: ' + (response.message || 'Error desconocido'), 'error');
                }
            }, 'json')
            .fail(function(jqXHR, textStatus, errorThrown) {
                console.error("AJAX Error:", textStatus, errorThrown, jqXHR.responseText);
                swal('Error de Conexión', 'No se pudo conectar con el servidor. Revisa la consola.', 'error');
            });
        }

        function renderGroupsInterface(data) {
            // 1. Render Students Without Group
            const listSinGrupo = $('#lista-sin-grupo');
            listSinGrupo.empty();
            $('#count-sin-grupo').text(data.sin_grupo.length);

            data.sin_grupo.forEach(est => {
                const card = createStudentCard(est);
                listSinGrupo.append(card);
            });

            // 2. Render Groups
            const gridGrupos = $('#grid-grupos');
            gridGrupos.empty();

            data.grupos.forEach(grupo => {
                const grupoCard = $(`
                    <div class="group-card droppable-zone" data-grupo-id="${grupo.id}" style="border: 1px solid #e2e8f0; border-radius: 8px; padding: 10px; background: #f8fafc; display: flex; flex-direction: column;">
                        <div style="display:flex; justify-content:space-between; align-items:center; border-bottom: 2px solid #ddd; padding-bottom: 5px; margin-bottom: 5px;">
                            <strong style="color: #333;">${grupo.nombre}</strong>
                            <div>
                                <span class="badge-info" style="font-size:0.75rem;">${grupo.estudiantes.length} / ${grupo.cupos || '∞'}</span>
                                <button onclick="deleteGrupo(${grupo.id})" style="border:none; background:none; color:#ef4444; cursor:pointer;"><i class='bx bx-trash'></i></button>
                            </div>
                        </div>
                        <div style="font-size: 0.8rem; color: #666; margin-bottom: 5px;">${grupo.horario || 'Sin horario'}</div>
                        
                        <div class="group-students-list" style="flex: 1; overflow-y: auto; background: white; border: 1px solid #eee; border-radius: 4px; padding: 5px;">
                            <!-- Students go here -->
                        </div>
                    </div>
                `);

                const studentContainer = grupoCard.find('.group-students-list');
                grupo.estudiantes.forEach(est => {
                    studentContainer.append(createStudentCard(est));
                });

                gridGrupos.append(grupoCard);
            });

            initDragAndDrop();
        }

        function createStudentCard(est) {
            const n = est.nombres || '?';
            const a = est.apellidos || '?';
            const initials = (n.charAt(0) + a.charAt(0)).toUpperCase();
            
            // Determine category color based on nombre_ciclo
            let categoryColor = '#94a3b8'; // default gray
            let categoryBg = '#f1f5f9';
            
            if(est.nombre_ciclo) {
                const ciclo = est.nombre_ciclo.toLowerCase();
                if(ciclo.includes('infan') || ciclo.includes('niñ')) {
                    categoryBg = '#fef3c7'; categoryColor = '#92400e'; // amber
                } else if(ciclo.includes('adolescen') || ciclo.includes('juven')) {
                    categoryBg = '#dbeafe'; categoryColor = '#1e40af'; // blue
                } else if(ciclo.includes('adult')) {
                    categoryBg = '#d1fae5'; categoryColor = '#065f46'; // green
                } else if(ciclo.includes('mayor') || ciclo.includes('ancian')) {
                    categoryBg = '#e0e7ff'; categoryColor = '#4338ca'; // indigo
                }
            }
            
            const categoryBadge = est.nombre_ciclo ? 
                `<span style="font-size: 0.7rem; padding: 2px 6px; background: ${categoryBg}; color: ${categoryColor}; border-radius: 3px; white-space: nowrap;">${est.nombre_ciclo}${est.edad ? ` (${est.edad})` : ''}</span>` : 
                '';
            
            const levelRaw = (est.nivel_desempeno || '').toString().trim().toUpperCase();
            let levelBg = '#e5e7eb';
            let levelColor = '#374151';
            let levelLabel = 'Sin diagnóstico';
            if(levelRaw === 'ALTO') {
                levelBg = '#fee2e2';
                levelColor = '#b91c1c';
                levelLabel = 'Nivel Alto';
            } else if(levelRaw === 'MEDIO') {
                levelBg = '#fef3c7';
                levelColor = '#92400e';
                levelLabel = 'Nivel Medio';
            } else if(levelRaw === 'BAJO') {
                levelBg = '#dcfce7';
                levelColor = '#166534';
                levelLabel = 'Nivel Bajo';
            }
            const levelBadge = `<span style="font-size: 0.7rem; padding: 2px 6px; background: ${levelBg}; color: ${levelColor}; border-radius: 3px; white-space: nowrap;"><i class='bx bx-line-chart'></i> ${levelLabel}</span>`;
            
            return $(`
                <div class="student-item" draggable="true" data-documento="${est.documento}" style="padding: 8px; margin-bottom: 5px; background: white; border: 1px solid #cbd5e1; border-radius: 4px; cursor: grab; display: flex; align-items: center; gap: 8px;">
                    <div style="width: 24px; height: 24px; background: #e0e7ff; color: #4338ca; border-radius: 50%; display: flex; align-items: center; justify-content: center; font-size: 0.7rem; font-weight: bold;">
                        ${initials}
                    </div>
                    <div style="display:flex; flex-direction:column; overflow:hidden; flex: 1;">
                        <span style="font-size: 0.85rem; font-weight: 600; color: #1f2937; white-space: nowrap; overflow: hidden; text-overflow: ellipsis;">${n} ${a}</span>
                        <div style="display: flex; gap: 6px; align-items: center;">
                            <span style="font-size: 0.75rem; color: #4b5563;">${est.documento}</span>
                            ${categoryBadge}
                        </div>
                        <div style="margin-top:4px;">
                            ${levelBadge}
                        </div>
                    </div>
                </div>
            `);
        }

        function initDragAndDrop() {
            const draggables = document.querySelectorAll('.student-item');
            const dropZones = document.querySelectorAll('.droppable-zone');

            draggables.forEach(draggable => {
                draggable.addEventListener('dragstart', () => {
                    draggable.classList.add('dragging');
                });

                draggable.addEventListener('dragend', () => {
                    draggable.classList.remove('dragging');
                });
            });

            dropZones.forEach(zone => {
                zone.addEventListener('dragover', e => {
                    e.preventDefault();
                    zone.style.background = '#e0f2fe'; // Highlight
                });

                zone.addEventListener('dragleave', () => {
                   // zone.style.background = zone.getAttribute('data-grupo-id') == '0' ? 'transparent' : '#f8fafc';
                   // Simpler: clear inline style if necessary or just reset class
                   zone.style.background = ''; 
                });

                zone.addEventListener('drop', e => {
                    e.preventDefault();
                    zone.style.background = '';
                    const draggable = document.querySelector('.dragging');
                    const targetGrupoId = zone.getAttribute('data-grupo-id'); // '0' for no group, or ID
                    
                    // Logic to assign
                    // 1. Visually move (optional, or reload)
                    // 2. Call Backend
                    const documento = draggable.getAttribute('data-documento');

                    // If dropped in created group zone, it might be nested in .group-students-list
                    // Actually, for groups, the drop zone is the whole card, but we append to the list
                    
                    // Find correct container to append
                    let appendTarget = zone;
                    if(zone.classList.contains('group-card')) {
                         appendTarget = zone.querySelector('.group-students-list');
                    }

                    appendTarget.appendChild(draggable);
                    
                    // Call API
                    assignStudentToGroup(documento, targetGrupoId);
                });
            });
        }

        function assignStudentToGroup(documento, grupoId) {
            $.post('ajax_registro.php', { 
                action: 'assign_group', 
                documento: documento, 
                id_curso: currentCurso.id, 
                id_grupo: grupoId 
            }, function(response) {
                if(!response.success) {
                    swal('Error', 'No se pudo mover el estudiante', 'error');
                    loadCursoData(currentCurso.id); // Revert UI
                } else {
                     swal({ 
                         title: 'Actualizado', 
                         text: 'Estudiante movido correctamente', 
                         type: 'success', 
                         timer: 1000, 
                         showConfirmButton: false 
                     });
                     // Update counts
                     loadCursoData(currentCurso.id); // Reload to ensure quotas/counts are correct
                }
            }, 'json');
        }

        // --- FORM HELPERS ---
        function submitForm(form, callback, modalId) {
            const data = form.serialize();
            $.post('ajax_registro.php', data, function(response) {
                if (response.success) {
                    swal('Éxito', response.message, 'success');
                    closeModal(modalId);
                    form[0].reset();
                    // Clear checkboxes manually if needed (reset() handles it but verify)
                    if (callback) callback();
                } else {
                    swal('Error', response.message, 'error');
                }
            }, 'json').fail(function() {
                swal('Error', 'Fallo de conexión', 'error');
            });
        }

        // --- MODAL HELPERS ---
        function openModal(modalId) {
            // Reset to "Add" mode
            if(modalId === 'modalEscuela') {
                $('#modalTitleEscuela').text('Nueva Escuela de Formación');
                $('#escuelaAction').val('add_escuela');
                $('#escuelaId').val('');
                $('#escuelaNombre').val('');
                $('#escuelaDescripcion').val('');
                $('#btnSubmitEscuela').text('Guardar Escuela');
            } else if(modalId === 'modalArea') {
                $('#modalTitleArea').text('Nueva Área');
                $('#areaAction').val('add_area');
                $('#areaId').val('');
                $('#areaNombre').val('');
                $('#btnSubmitArea').text('Guardar Área');
            } else if(modalId === 'modalCurso') {
                $('#modalTitleCurso').text('Nuevo Curso');
                $('#cursoAction').val('add_curso');
                $('#cursoId').val('');
                $('#cursoNombre').val('');
                $('#cursoProfesor').val('');
                $('#cursoTitulado').val('no');
                $('#cursoCupos').val('20');
                $('#cursoDescripcion').val('');
                $('#cursoHorario').val('');
                $('#horaInicio').val('');
                $('#horaFin').val('');
                $('input[name="dias[]"]').prop('checked', false);
                $('#btnSubmitCurso').text('Guardar Curso');
            } else if(modalId === 'modalGrupo') {
                $('#modalTitleGrupo').text('Nuevo Grupo');
                $('#grupoAction').val('add_grupo');
                $('#grupoId').val('');
                $('#grupoCursoId').val(currentCurso ? currentCurso.id : '');
                $('#grupoNombre').val('');
                $('#grupoHorario').val('');
                $('#horaInicioGrupo').val('');
                $('#horaFinGrupo').val('');
                $('input[name="dias_grupo[]"]').prop('checked', false);
                $('#grupoCupos').val('');
                $('#btnSubmitGrupo').text('Guardar Grupo');
                
                // Calculate and show quota information
                if(currentCurso && currentCurso.cupos > 0) {
                    // Get all existing groups for this course
                    $.post('ajax_registro.php', { action: 'get_grupos', id_curso: currentCurso.id }, function(response) {
                        if (response.success) {
                            let totalAsignada = 0;
                            response.data.forEach(grupo => {
                                totalAsignada += parseInt(grupo.cupos) || 0;
                            });
                            
                            const disponible = currentCurso.cupos - totalAsignada;
                            
                            $('#quotaCursoTotal').text(currentCurso.cupos);
                            $('#quotaAsignada').text(totalAsignada);
                            $('#quotaDisponible').text(disponible);
                            
                            // Set max attribute for input
                            $('#grupoCupos').attr('max', disponible);
                            
                            // Show/hide quota info
                            if(disponible <= 0) {
                                $('#quotaInfo').show();
                                $('#quotaDisponible').parent().html('• <span style="color: #b91c1c;">Disponibles: <strong>0</strong> (Curso lleno)</span>');
                                swal('Atención', 'Este curso ya tiene todos sus cupos asignados a grupos.', 'warning');
                            } else {
                                $('#quotaInfo').show();
                            }
                        }
                    }, 'json');
                } else {
                    $('#quotaInfo').hide();
                }
            }
            $('#' + modalId).fadeIn();
        }

        function openEditEscuela(event, id, nombre, descripcion) {
            event.stopPropagation();
            $('#modalTitleEscuela').text('Editar Escuela');
            $('#escuelaAction').val('edit_escuela');
            $('#escuelaId').val(id);
            $('#escuelaNombre').val(nombre);
            $('#escuelaDescripcion').val(descripcion);
            $('#btnSubmitEscuela').text('Actualizar Escuela');
            $('#modalEscuela').fadeIn();
        }

        function openEditArea(event, id, nombre) {
            event.stopPropagation();
            $('#modalTitleArea').text('Editar Área');
            $('#areaAction').val('edit_area');
            $('#areaId').val(id);
            $('#areaNombre').val(nombre);
            $('#btnSubmitArea').text('Actualizar Área');
            $('#modalArea').fadeIn();
        }
        
        function openEditCurso(event, item) {
            event.stopPropagation();
            $('#modalTitleCurso').text('Editar Curso');
            $('#cursoAction').val('edit_curso');
            $('#cursoId').val(item.idcurso);
            $('#cursoNombre').val(item.nombre);
            $('#cursoProfesor').val(item.quien_lo_da);
            $('#cursoTitulado').val(item.es_titulado);
            $('#cursoDescripcion').val(item.en_que || '');
            $('#cursoCupos').val(item.cupos || 0);
            $('#cursoHorario').val(item.horario_clase || '');
            
            // Parse Time for Inputs
            $('#horaInicio').val('');
            $('#horaFin').val('');
            if(item.horario_clase) {
                const parts = item.horario_clase.split(' - ');
                if(parts[0]) $('#horaInicio').val(convertTo24h(parts[0]));
                if(parts[1]) $('#horaFin').val(convertTo24h(parts[1]));
            }

            // Set Checkboxes
            $('input[name="dias[]"]').prop('checked', false);
            if(item.dias_clase) {
                const diasArray = item.dias_clase.split(', ');
                diasArray.forEach(dia => {
                    $(`input[name="dias[]"][value="${dia}"]`).prop('checked', true);
                });
            }

            $('#btnSubmitCurso').text('Actualizar Curso');
            $('#modalCurso').fadeIn();
        }

        function closeModal(modalId) {
            $('#' + modalId).fadeOut();
        }

        // Close modal when clicking outside
        $(window).on('click', function(event) {
            if ($(event.target).hasClass('modal')) {
                $('.modal').fadeOut();
            }
        });

        // --- SELECT CURSO (for enrollment report) ---
        function selectCurso(element, id, nombre) {
            // UI Update
            $('#list-cursos .list-item').removeClass('active');
            $(element).addClass('active');

            // Load Enrollment Report for Curso
            loadInscritos('curso', id, nombre);
        }

        // --- LOAD ENROLLMENT REPORT ---
        function loadInscritos(tipo, id, nombre) {
            // Show overlay and loading state
            $('#overlay-reporte').hide();
            $('#reporte-info').show();
            $('#list-inscritos').html('<li style="text-align:center;color:#999;padding:20px;">Cargando...</li>');
            
            // Update header
            const tipoLabel = tipo === 'escuela' ? 'Escuela:' : (tipo === 'area' ? 'Área:' : 'Curso:');
            $('#reporte-tipo').text(tipoLabel);
            $('#reporte-nombre').text(nombre);
            
            // Make AJAX request
            $.post('ajax_registro.php', {
                action: 'get_inscritos',
                tipo: tipo,
                id: id
            }, function(response) {
                if (response.success) {
                    const data = response.data;
                    const lista = $('#list-inscritos');
                    lista.empty();
                    
                    // Update count
                    $('#count-inscritos').text(data.total);
                    $('#reporte-total').text(data.total);
                    
                    if (data.total === 0) {
                        lista.append('<li style="text-align:center;color:#ccc;padding:20px;">No hay estudiantes inscritos</li>');
                    } else {
                        // Display each student
                        data.estudiantes.forEach((estudiante, index) => {
                            // Ciclo vital badge color
                            const cicloColor = estudiante.ciclo_vital ? '#10b981' : '#999';
                            
                            lista.append(`
                                <li class="list-item" style="cursor: default;">
                                    <div style="display: flex; align-items: center; gap: 10px;">
                                        <div style="background: #e0e7ff; color: #4338ca; width: 30px; height: 30px; border-radius: 50%; display: flex; align-items: center; justify-content: center; font-weight: bold; font-size: 0.9em;">
                                            ${index + 1}
                                        </div>
                                        <div style="flex: 1;">
                                            <span class="list-item-title">${estudiante.nombre}</span>
                                            <span class="list-item-subtitle" style="display: block;">Doc: ${estudiante.documento}</span>
                                            <div style="display:flex; flex-direction:column; gap:2px; margin-top:4px;">
                                                ${estudiante.ciclo_vital !== 'N/A' ? `<small style="background: ${cicloColor}; color: white; padding: 2px 6px; border-radius: 4px; font-size: 0.7em; display: inline-block; width: fit-content;"><i class='bx bx-user-circle'></i> ${estudiante.ciclo_vital}</small>` : ''}
                                                ${estudiante.nivel_desempeno !== 'N/A' 
                                                    ? `<small style="background: #3b82f6; color: white; padding: 2px 6px; border-radius: 4px; font-size: 0.7em; display: inline-block; width: fit-content;"><i class='bx bx-trending-up'></i> ${estudiante.nivel_desempeno}</small>` 
                                                    : `<small style="background: #9ca3af; color: white; padding: 2px 6px; border-radius: 4px; font-size: 0.7em; display: inline-block; width: fit-content;"><i class='bx bx-help-circle'></i> Sin Evaluar</small>`}
                                                ${estudiante.curso_nombre ? `<small style="color: #888; font-size: 0.75em; display: block;"><i class='bx bx-book'></i> ${estudiante.curso_nombre}</small>` : ''}
                                            </div>
                                        </div>
                                    </div>
                                </li>
                            `);
                        });
                    }
                } else {
                    $('#list-inscritos').html('<li style="text-align:center;color:#e74c3c;padding:20px;">Error al cargar datos</li>');
                }
            }, 'json').fail(function() {
                $('#list-inscritos').html('<li style="text-align:center;color:#e74c3c;padding:20px;">Error de conexión</li>');
            });
        }
    </script>
</body>
</html>