<?php
session_start();

require '../../backend/bd/Conexion.php';
require '../../backend/fpdf/fpdf.php';

$scope = $_GET['scope'] ?? null;
$id = isset($_GET['id']) ? (int)$_GET['id'] : 0;

if(!$scope || $id <= 0) {
    die('Parámetros inválidos');
}

function convertText($text) {
    if($text === null) {
        return '';
    }
    return mb_convert_encoding($text, 'ISO-8859-1', 'UTF-8');
}

class CursoPDF extends FPDF {
    public $reportTitle = '';
    public $reportSubtitle = '';

    function Header() {
        $this->Image('../../backend/img/ico.png', 12, 5, 20);
        $this->SetFont('Arial', 'B', 14);
        $this->Cell(0, 8, convertText('EFAC - Reporte de Estudiantes'), 0, 1, 'C');
        $this->SetFont('Arial', '', 11);
        $this->Cell(0, 6, convertText($this->reportTitle), 0, 1, 'C');
        if(!empty($this->reportSubtitle)) {
            $this->SetFont('Arial', 'I', 10);
            $this->Cell(0, 6, convertText($this->reportSubtitle), 0, 1, 'C');
        }
        $this->Ln(2);
    }

    function Footer() {
        $this->SetY(-15);
        $this->SetFont('Arial', 'I', 8);
        $this->Cell(0, 10, convertText('Página ' . $this->PageNo() . '/{nb}'), 0, 0, 'C');
    }

    function TableHeader() {
        $this->SetFillColor(99, 102, 241);
        $this->SetTextColor(255, 255, 255);
        $this->SetFont('Arial', 'B', 9);
        $this->Cell(12, 8, convertText('#'), 1, 0, 'C', true);
        $this->Cell(65, 8, convertText('Nombre completo'), 1, 0, 'L', true);
        $this->Cell(32, 8, convertText('Documento'), 1, 0, 'L', true);
        $this->Cell(40, 8, convertText('Curso / Grupo'), 1, 0, 'L', true);
        $this->Cell(35, 8, convertText('Nivel diagnóstico'), 1, 0, 'L', true);
        $this->Cell(22, 8, convertText('Ciclo'), 1, 1, 'L', true);
        $this->SetTextColor(0);
    }

    function TableRow($index, $row) {
        $this->SetFont('Arial', '', 8.5);
        $this->Cell(12, 7, convertText($index), 1, 0, 'C');
        $this->Cell(65, 7, convertText($row['nombre']), 1, 0, 'L');
        $this->Cell(32, 7, convertText($row['documento']), 1, 0, 'L');
        $cursoInfo = $row['curso'] ?? '';
        if(!empty($row['grupo'])) {
            $cursoInfo .= ($cursoInfo ? ' / ' : '') . $row['grupo'];
        }
        $this->Cell(40, 7, convertText($cursoInfo ?: '-'), 1, 0, 'L');
        $nivel = strtoupper($row['nivel'] ?? '');
        $this->Cell(35, 7, convertText($nivel ?: 'Sin registro'), 1, 0, 'L');
        $this->Cell(22, 7, convertText($row['ciclo'] ?? '-'), 1, 1, 'L');
    }
}

function fetchMetadata(PDO $connect, string $scope, int $id): array {
    switch($scope) {
        case 'escuela':
            $stmt = $connect->prepare("SELECT nombre FROM escuelas_formacion WHERE id = ?");
            $stmt->execute([$id]);
            $row = $stmt->fetch(PDO::FETCH_ASSOC);
            return ['title' => 'Escuela: ' . ($row['nombre'] ?? 'Desconocida'), 'subtitle' => ''];
        case 'area':
            $stmt = $connect->prepare("SELECT a.nombre, e.nombre AS escuela FROM areas a LEFT JOIN escuelas_formacion e ON a.id_escuela = e.id WHERE a.id = ?");
            $stmt->execute([$id]);
            $row = $stmt->fetch(PDO::FETCH_ASSOC);
            return ['title' => 'Área: ' . ($row['nombre'] ?? 'Desconocida'), 'subtitle' => 'Escuela: ' . ($row['escuela'] ?? 'N/D')];
        case 'curso':
            $stmt = $connect->prepare("SELECT c.nombre, a.nombre AS area, e.nombre AS escuela FROM cursos c LEFT JOIN areas a ON c.id_area = a.id LEFT JOIN escuelas_formacion e ON a.id_escuela = e.id WHERE c.idcurso = ?");
            $stmt->execute([$id]);
            $row = $stmt->fetch(PDO::FETCH_ASSOC);
            return [
                'title' => 'Curso: ' . ($row['nombre'] ?? 'Desconocido'),
                'subtitle' => trim('Área: ' . ($row['area'] ?? 'N/D') . ' | Escuela: ' . ($row['escuela'] ?? 'N/D'))
            ];
        case 'grupo':
            $stmt = $connect->prepare("SELECT g.nombre, c.nombre AS curso, a.nombre AS area FROM cursos_grupos g LEFT JOIN cursos c ON g.id_curso = c.idcurso LEFT JOIN areas a ON c.id_area = a.id WHERE g.id = ?");
            $stmt->execute([$id]);
            $row = $stmt->fetch(PDO::FETCH_ASSOC);
            return [
                'title' => 'Grupo: ' . ($row['nombre'] ?? 'Desconocido'),
                'subtitle' => trim('Curso: ' . ($row['curso'] ?? 'N/D') . ' | Área: ' . ($row['area'] ?? 'N/D'))
            ];
        default:
            return ['title' => 'Reporte', 'subtitle' => ''];
    }
}

function buildStudentsQuery(string $scope): array {
    $base = "
        SELECT DISTINCT 
            TRIM(CONCAT(COALESCE(re.nombres, u.name, p.nompa), ' ', COALESCE(re.apellidos, '', p.apepa))) AS nombre,
            i.documento,
            c.nombre AS curso,
            g.nombre AS grupo,
            vr.niveldesempeño AS nivel,
            cv.nombre_ciclo AS ciclo
        FROM inscripciones i
        LEFT JOIN cursos c ON i.idcurso = c.idcurso
        LEFT JOIN cursos_grupos g ON i.id_grupo = g.id
        LEFT JOIN users u ON i.documento = u.username OR i.documento = u.email
        LEFT JOIN registro_estudiantes re ON u.id = re.user_id
        LEFT JOIN patients p ON i.documento = p.numhs
        LEFT JOIN ciclo_vital cv ON TIMESTAMPDIFF(YEAR, re.fecha_nacimiento, CURDATE()) BETWEEN cv.min_edad AND cv.max_edad
        LEFT JOIN valoracionriesgo vr ON re.id = vr.id_estudiante
            AND vr.id_valoracion = (
                SELECT vr2.id_valoracion
                FROM valoracionriesgo vr2
                WHERE vr2.id_estudiante = re.id
                ORDER BY vr2.fere DESC
                LIMIT 1
            )
        ";

    switch($scope) {
        case 'curso':
            return [$base . " WHERE i.idcurso = ? AND i.state = '1' ORDER BY nombre", 'params' => ['idcurso']];
        case 'area':
            return [$base . " WHERE c.id_area = ? AND i.state = '1' ORDER BY nombre", 'params' => ['area']];
        case 'escuela':
            $base .= " LEFT JOIN areas a ON c.id_area = a.id ";
            return [$base . " WHERE a.id_escuela = ? AND i.state = '1' ORDER BY nombre", 'params' => ['escuela']];
        case 'grupo':
            return [$base . " WHERE i.id_grupo = ? AND i.state = '1' ORDER BY nombre", 'params' => ['grupo']];
        default:
            return [$base . " WHERE 1=0", 'params' => []];
    }
}

function fetchStudents(PDO $connect, string $scope, int $id): array {
    list($sql,) = buildStudentsQuery($scope);
    $stmt = $connect->prepare($sql);
    $stmt->execute([$id]);
    $rows = $stmt->fetchAll(PDO::FETCH_ASSOC);
    return array_map(function($row) {
        $row['nombre'] = $row['nombre'] ?: 'N/D';
        $row['documento'] = $row['documento'] ?: '-';
        $row['nivel'] = $row['nivel'] ?: 'Sin registro';
        $row['ciclo'] = $row['ciclo'] ?: '-';
        return $row;
    }, $rows);
}

$metadata = fetchMetadata($connect, $scope, $id);
$students = fetchStudents($connect, $scope, $id);

$pdf = new CursoPDF('L', 'mm', 'A4');
$pdf->AliasNbPages();
$pdf->reportTitle = $metadata['title'];
$pdf->reportSubtitle = $metadata['subtitle'];
$pdf->AddPage();

$pdf->SetFont('Arial', '', 10);
$pdf->Ln(4);
$pdf->Cell(0, 6, convertText('Total de estudiantes: ' . count($students)), 0, 1, 'L');
$pdf->Ln(2);

$pdf->TableHeader();

if(empty($students)) {
    $pdf->Cell(206, 8, convertText('No se encontraron estudiantes inscritos para este filtro.'), 1, 1, 'C');
} else {
    $index = 1;
    foreach($students as $student) {
        $pdf->TableRow($index++, $student);
    }
}

$pdf->Output('I', 'reporte_estudiantes.pdf');
exit;
?>
