<?php
session_start();

// Incluir el archivo de conexión
require_once 'conexion.php';
// Incluir configuración de email
require_once 'email_config.php';

// Variables para la lista de abuelos
$lista_abuelos = [];
$mostrar_mapa = false;
$latitud_busqueda = "";
$longitud_busqueda = "";
$nombre_busqueda = "";
$abuelo_seleccionado = "";
$tiempo_filtro = "2"; // Valor predeterminado: 2 horas

// Verificar si se accede con cédula desde QR (para registro automático)
$cedula_qr = isset($_GET['cedula']) ? trim($_GET['cedula']) : '';
$registro_automatico = !empty($cedula_qr);

// Procesar registro automático de ubicación desde QR
if ($registro_automatico && $_SERVER["REQUEST_METHOD"] == "POST" && isset($_POST['auto_register'])) {
    $cedula = trim($_POST['cedula']);
    $latitud = floatval($_POST['latitud']);
    $longitud = floatval($_POST['longitud']);
    
    if ($latitud >= -90 && $latitud <= 90 && $longitud >= -180 && $longitud <= 180) {
        // Buscar paciente por cédula
        $sql = "SELECT numhs, nompa, apepa, correo FROM patients WHERE numhs = :cedula";
        $stmt = $conexion->prepare($sql);
        $stmt->execute(['cedula' => $cedula]);
        
        if ($paciente = $stmt->fetch(PDO::FETCH_ASSOC)) {
            // Insertar ubicación automáticamente
            $stmt = $conexion->prepare("INSERT INTO ubicaciones_completas (nompa, documento, correo, latitud, longitud) VALUES (:nompa, :documento, :correo, :latitud, :longitud)");
            
            if ($stmt->execute([
                'nompa' => $paciente['nompa'],
                'documento' => $paciente['numhs'],
                'correo' => $paciente['correo'],
                'latitud' => $latitud,
                'longitud' => $longitud
            ])) {
                // Enviar correo de alerta
                $nombre_completo = $paciente['nompa'] . ' ' . $paciente['apepa'];
                $fecha_hora = date('d/m/Y H:i:s');
                
                // Correo de destino fijo para alertas
                $correo_destino = 'adultomayor@sibate-cundinamarca.gov.co';
                
                // Enviar correo al destinatario fijo
                $correo_enviado = enviarCorreoUbicacion(
                    $correo_destino,
                    $nombre_completo,
                    $latitud,
                    $longitud,
                    $fecha_hora
                );
                
                // Log del resultado
                if ($correo_enviado) {
                    error_log("Correo de alerta enviado a: " . $correo_destino);
                } else {
                    error_log("No se pudo enviar correo de alerta a: " . $correo_destino);
                }
                
                $_SESSION['registro_exitoso'] = "Ubicación registrada correctamente para: {$paciente['nompa']} {$paciente['apepa']}";
                $_SESSION['ubicacion_lat'] = $latitud;
                $_SESSION['ubicacion_lng'] = $longitud;
                $_SESSION['ubicacion_nombre'] = $paciente['nompa'] . ' ' . $paciente['apepa'];
                $_SESSION['fecha_registro'] = $fecha_hora;
                $_SESSION['enviar_whatsapp'] = true; // Activar envío de WhatsApp
                header("Location: ubicar_abuelos.php?success=1");
                exit;
            }
        }
    }
}

// Procesar búsqueda de abuelos
if ($_SERVER["REQUEST_METHOD"] == "POST") {
    // Verificar si se cambió el filtro de tiempo
    if (isset($_POST['tiempo_filtro'])) {
        $tiempo_filtro = $_POST['tiempo_filtro'];
    }
    
    // Procesar búsqueda por selección
    if (isset($_POST['buscar_abuelo']) && !empty($_POST['abuelo_seleccionado'])) {
        $abuelo_seleccionado = $_POST['abuelo_seleccionado'];
        
        // Consulta para obtener la ubicación
        $sql_ubicacion = "SELECT u.latitud, u.longitud, p.nompa, p.apepa FROM ubicaciones_completas u
                         JOIN patients p ON u.documento = p.numhs
                         WHERE u.id = :id";
        
        $stmt = $conexion->prepare($sql_ubicacion);
        $stmt->execute(['id' => $abuelo_seleccionado]);
        
        if ($ubicacion = $stmt->fetch(PDO::FETCH_ASSOC)) {
            $mostrar_mapa = true;
            $latitud_busqueda = $ubicacion['latitud'];
            $longitud_busqueda = $ubicacion['longitud'];
            $nombre_busqueda = $ubicacion['nompa'] . ' ' . $ubicacion['apepa'];
        }
    }
}

// Obtener lista de abuelos registrados en el período seleccionado
$sql_abuelos = "SELECT u.id, u.documento, u.latitud, u.longitud, p.nompa, p.apepa, u.fecha_registro 
                FROM ubicaciones_completas u
                JOIN patients p ON u.documento = p.numhs
                WHERE u.fecha_registro >= NOW() - INTERVAL :horas HOUR
                ORDER BY u.fecha_registro DESC";

$stmt = $conexion->prepare($sql_abuelos);
$stmt->execute(['horas' => $tiempo_filtro]);
$lista_abuelos = $stmt->fetchAll(PDO::FETCH_ASSOC);

// Obtener todos los registros para el mapa completo
$todas_ubicaciones = [];
if (isset($_POST['ver_todas']) || isset($_POST['ver_todas_mapa'])) {
    $sql_todas = "SELECT u.id, u.latitud, u.longitud, p.nompa, p.apepa, u.fecha_registro 
                  FROM ubicaciones_completas u
                  JOIN patients p ON u.documento = p.numhs
                  WHERE u.fecha_registro >= NOW() - INTERVAL :horas HOUR
                  ORDER BY u.fecha_registro DESC";
    
    $stmt = $conexion->prepare($sql_todas);
    $stmt->execute(['horas' => $tiempo_filtro]);
    $todas_ubicaciones = $stmt->fetchAll(PDO::FETCH_ASSOC);
    
    // Si es solo para ver todas en el mapa
    if (isset($_POST['ver_todas_mapa'])) {
        $mostrar_mapa = true;
    }
}
?>

<!DOCTYPE html>
<html lang="es">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>Ubicar Personas - Hogar de los Abuelos</title>
    <link href="https://cdn.jsdelivr.net/npm/bootstrap@5.3.0/dist/css/bootstrap.min.css" rel="stylesheet">
    <link rel="stylesheet" href="https://unpkg.com/leaflet@1.9.4/dist/leaflet.css" />
    <link href='https://unpkg.com/boxicons@2.1.4/css/boxicons.min.css' rel='stylesheet'>
    <style>
        :root {
            --primary-color: #4A90E2;
            --secondary-color: #667eea;
            --success-color: #28a745;
            --danger-color: #dc3545;
            --warning-color: #ffc107;
            --dark-color: #2c3e50;
            --light-bg: #f8f9fa;
        }
        
        body {
            font-family: 'Segoe UI', Tahoma, Geneva, Verdana, sans-serif;
            background: linear-gradient(135deg, #667eea 0%, #764ba2 100%);
            min-height: 100vh;
            padding-bottom: 50px;
        }
        
        .main-header {
            background: white;
            padding: 25px 0;
            box-shadow: 0 4px 20px rgba(0,0,0,0.1);
            margin-bottom: 40px;
            border-radius: 0 0 20px 20px;
        }
        
        .main-header h1 {
            color: var(--dark-color);
            font-weight: 700;
            display: flex;
            align-items: center;
            justify-content: center;
            gap: 15px;
            margin: 0;
            font-size: 2.2rem;
        }
        
        .main-header h1 i {
            font-size: 2.5rem;
            color: var(--primary-color);
        }
        
        .content-wrapper {
            max-width: 1200px;
            margin: 0 auto;
            padding: 0 20px;
        }
        
        .info-card {
            background: white;
            border-radius: 15px;
            padding: 30px;
            box-shadow: 0 10px 30px rgba(0,0,0,0.15);
            margin-bottom: 30px;
            transition: transform 0.3s ease, box-shadow 0.3s ease;
        }
        
        .info-card:hover {
            transform: translateY(-5px);
            box-shadow: 0 15px 40px rgba(0,0,0,0.2);
        }
        
        .card-header-custom {
            display: flex;
            align-items: center;
            gap: 12px;
            margin-bottom: 25px;
            padding-bottom: 15px;
            border-bottom: 2px solid #e9ecef;
        }
        
        .card-header-custom i {
            font-size: 2rem;
            color: var(--primary-color);
        }
        
        .card-header-custom h3 {
            margin: 0;
            color: var(--dark-color);
            font-weight: 600;
            font-size: 1.5rem;
        }
        
        #mapa {
            height: 500px;
            width: 100%;
            border-radius: 15px;
            box-shadow: 0 5px 20px rgba(0,0,0,0.15);
            border: 3px solid white;
        }
        
        .map-container {
            margin-top: 20px;
        }
        
        .btn-primary-custom {
            background: linear-gradient(135deg, var(--primary-color) 0%, var(--secondary-color) 100%);
            border: none;
            padding: 12px 30px;
            font-weight: 600;
            border-radius: 10px;
            transition: all 0.3s ease;
            box-shadow: 0 4px 15px rgba(102, 126, 234, 0.4);
        }
        
        .btn-primary-custom:hover {
            transform: translateY(-2px);
            box-shadow: 0 6px 20px rgba(102, 126, 234, 0.6);
        }
        
        .btn-success-custom {
            background: linear-gradient(135deg, #28a745 0%, #20c997 100%);
            border: none;
            padding: 12px 30px;
            font-weight: 600;
            border-radius: 10px;
            transition: all 0.3s ease;
            box-shadow: 0 4px 15px rgba(40, 167, 69, 0.4);
            color: white;
            text-decoration: none;
            display: inline-flex;
            align-items: center;
            gap: 8px;
        }
        
        .btn-success-custom:hover {
            transform: translateY(-2px);
            box-shadow: 0 6px 20px rgba(40, 167, 69, 0.6);
            color: white;
        }
        
        .alert-custom {
            border-radius: 12px;
            border: none;
            padding: 20px;
            box-shadow: 0 4px 15px rgba(0,0,0,0.1);
            display: flex;
            align-items: center;
            gap: 15px;
        }
        
        .alert-custom i {
            font-size: 2rem;
        }
        
        .ubicacion-info {
            background: linear-gradient(135deg, #667eea 0%, #764ba2 100%);
            color: white;
            padding: 20px;
            border-radius: 12px;
            margin-bottom: 20px;
            box-shadow: 0 4px 15px rgba(0,0,0,0.2);
        }
        
        .ubicacion-info h4 {
            margin: 0;
            font-weight: 600;
            display: flex;
            align-items: center;
            gap: 10px;
        }
        
        .ubicacion-info h4 i {
            font-size: 1.8rem;
        }
        
        /* Animación de carga para el mapa */
        .map-loading {
            position: absolute;
            top: 50%;
            left: 50%;
            transform: translate(-50%, -50%);
            z-index: 1000;
        }
        
        .spinner {
            border: 4px solid rgba(255, 255, 255, 0.3);
            border-radius: 50%;
            border-top: 4px solid var(--primary-color);
            width: 40px;
            height: 40px;
            animation: spin 1s linear infinite;
        }
        
        @keyframes spin {
            0% { transform: rotate(0deg); }
            100% { transform: rotate(360deg); }
        }
        
        /* Efecto de aparición para las cards */
        .info-card {
            animation: slideUp 0.5s ease-out;
        }
        
        @keyframes slideUp {
            from {
                opacity: 0;
                transform: translateY(30px);
            }
            to {
                opacity: 1;
                transform: translateY(0);
            }
        }
        
        /* Estilos para el mensaje de agradecimiento */
        .mensaje-agradecimiento {
            position: fixed;
            top: 50%;
            left: 50%;
            transform: translate(-50%, -50%);
            background: linear-gradient(135deg, #667eea 0%, #764ba2 100%);
            padding: 40px;
            border-radius: 20px;
            box-shadow: 0 20px 60px rgba(0, 0, 0, 0.3);
            z-index: 9999;
            max-width: 500px;
            width: 90%;
            animation: fadeInScale 0.5s ease-out;
        }
        
        .mensaje-agradecimiento-overlay {
            position: fixed;
            top: 0;
            left: 0;
            width: 100%;
            height: 100%;
            background: rgba(0, 0, 0, 0.7);
            z-index: 9998;
            animation: fadeIn 0.5s ease-out;
        }
        
        .mensaje-agradecimiento h2 {
            color: white;
            text-align: center;
            margin-bottom: 20px;
            font-size: 2rem;
            font-weight: bold;
        }
        
        .mensaje-agradecimiento p {
            color: white;
            text-align: center;
            font-size: 1.2rem;
            line-height: 1.8;
            margin-bottom: 30px;
        }
        
        .mensaje-agradecimiento .emoji {
            font-size: 3rem;
            text-align: center;
            margin-bottom: 20px;
        }
        
        .mensaje-agradecimiento .btn-cerrar {
            display: block;
            width: 100%;
            padding: 15px;
            background: white;
            color: #667eea;
            border: none;
            border-radius: 10px;
            font-size: 1.1rem;
            font-weight: bold;
            cursor: pointer;
            transition: all 0.3s ease;
        }
        
        .mensaje-agradecimiento .btn-cerrar:hover {
            transform: scale(1.05);
            box-shadow: 0 5px 15px rgba(255, 255, 255, 0.3);
        }
        
        .mensaje-agradecimiento .btn-whatsapp {
            display: block;
            width: 100%;
            padding: 15px 20px;
            background: #25D366;
            color: white;
            border: none;
            border-radius: 10px;
            font-size: 1.1rem;
            font-weight: bold;
            cursor: pointer;
            transition: all 0.3s ease;
            box-shadow: 0 4px 15px rgba(37, 211, 102, 0.4);
        }
        
        .mensaje-agradecimiento .btn-whatsapp:hover {
            transform: scale(1.05);
            background: #128C7E;
            box-shadow: 0 6px 20px rgba(37, 211, 102, 0.6);
        }
        
        .mensaje-agradecimiento .btn-whatsapp:active {
            transform: scale(0.98);
        }
        
        @keyframes fadeIn {
            from {
                opacity: 0;
            }
            to {
                opacity: 1;
            }
        }
        
        @keyframes fadeInScale {
            from {
                opacity: 0;
                transform: translate(-50%, -50%) scale(0.8);
            }
            to {
                opacity: 1;
                transform: translate(-50%, -50%) scale(1);
            }
        }
        @media (max-width: 768px) {
            .main-header h1 {
                font-size: 1.6rem;
            }
            
            .main-header h1 i {
                font-size: 2rem;
            }
            
            .info-card {
                padding: 20px;
            }
            
            .card-header-custom h3 {
                font-size: 1.2rem;
            }
            
            #mapa {
                height: 350px;
            }
            
            .btn-primary-custom, .btn-success-custom {
                padding: 10px 20px;
                font-size: 0.9rem;
            }
        }
        
        @media (max-width: 480px) {
            .main-header {
                padding: 15px 0;
            }
            
            .main-header h1 {
                font-size: 1.3rem;
                flex-direction: column;
                gap: 8px;
            }
            
            .info-card {
                padding: 15px;
            }
            
            .card-header-custom {
                flex-direction: column;
                align-items: flex-start;
                gap: 8px;
            }
            
            #mapa {
                height: 300px;
            }
            
            .btn-primary-custom, .btn-success-custom {
                padding: 10px 15px;
                font-size: 0.85rem;
                width: 100%;
            }
        }
    </style>
</head>
<body>
    <!-- Header -->
    <div class="main-header">
        <div class="container">
            <h1>
                <i class='bx bx-map-pin'></i>
                Ubicar Personas
            </h1>
        </div>
    </div>
    
    <!-- Main Content -->
    <div class="content-wrapper">
        
        <?php if (isset($_GET['success']) && isset($_SESSION['registro_exitoso'])): ?>
            <!-- Overlay -->
            <div class="mensaje-agradecimiento-overlay" id="mensajeOverlay"></div>
            
            <!-- Mensaje de Agradecimiento -->
            <div class="mensaje-agradecimiento" id="mensajeAgradecimiento">
                <div class="emoji">🙏💐</div>
                <h2>¡Muchas Gracias!</h2>
                <p>
                    Muchas gracias por tu ayuda.<br>
                    El Hogar de los Abuelos y mi familia te agradecen de corazón por tu apoyo y dedicación.<br>
                    Tu colaboración significa mucho para nosotros.
                </p>
                <div style="margin: 25px 0; padding: 20px; background: rgba(255,255,255,0.2); border-radius: 10px;">
                    <p style="font-size: 1.1rem; margin-bottom: 15px; font-weight: 600;">
                        📍 Ayuda al abuelo compartiendo su ubicación
                    </p>
                    <button id="btnEnviarWhatsApp" class="btn-whatsapp" onclick="enviarWhatsApp()">
                        <i class='bx bxl-whatsapp' style="font-size: 24px; vertical-align: middle;"></i>
                        Enviar Ubicación por WhatsApp
                    </button>
                </div>
                <button class="btn-cerrar" onclick="cerrarMensaje()">Cerrar</button>
            </div>
            <?php 
            $enviar_whatsapp = $_SESSION['enviar_whatsapp'] ?? false;
            $fecha_registro = $_SESSION['fecha_registro'] ?? date('d/m/Y H:i:s');
            unset($_SESSION['registro_exitoso']);
            unset($_SESSION['enviar_whatsapp']);
            unset($_SESSION['fecha_registro']);
            $mostrar_mapa = true;
            $latitud_busqueda = $_SESSION['ubicacion_lat'] ?? '';
            $longitud_busqueda = $_SESSION['ubicacion_lng'] ?? '';
            $nombre_busqueda = $_SESSION['ubicacion_nombre'] ?? '';
            ?>
        <?php endif; ?>
        
        <!-- Mapa Section -->
        <?php if ($mostrar_mapa): ?>
            <div class="info-card">
                <?php if (!isset($_POST['ver_todas_mapa']) && !empty($nombre_busqueda)): ?>
                    <div class="ubicacion-info">
                        <h4>
                            <i class='bx bx-user-pin'></i>
                            Ubicación de <?php echo htmlspecialchars($nombre_busqueda); ?>
                        </h4>
                    </div>
                <?php elseif (isset($_POST['ver_todas_mapa'])): ?>
                    <div class="ubicacion-info">
                        <h4>
                            <i class='bx bx-map-alt'></i>
                            Todas las ubicaciones (<?php echo count($todas_ubicaciones); ?>)
                        </h4>
                    </div>
                <?php endif; ?>
                
                <div class="map-container">
                    <div id="mapa"></div>
                </div>
                
                <?php if (!isset($_POST['ver_todas_mapa']) && !empty($latitud_busqueda)): ?>
                    <div class="text-center mt-4">
                        <a href="https://www.google.com/maps?q=<?php echo $latitud_busqueda; ?>,<?php echo $longitud_busqueda; ?>" 
                           target="_blank" class="btn-success-custom">
                            <i class='bx bx-map'></i>
                            Abrir en Google Maps
                        </a>
                    </div>
                <?php endif; ?>
            </div>
        <?php else: ?>
            <div class="info-card">
                <div class="alert-custom alert-info">
                    <i class='bx bx-info-circle'></i>
                    <div>
                        <strong>Información</strong><br>
                        La ubicación se mostrará automáticamente después de escanear el código QR.
                    </div>
                </div>
            </div>
        <?php endif; ?>
    </div>

<script src="https://unpkg.com/leaflet@1.9.4/dist/leaflet.js"></script>
<script>
<?php if ($mostrar_mapa): ?>
    // Inicializar el mapa
    var mapa = L.map('mapa');
    
    // Añadir capa de OpenStreetMap
    L.tileLayer('https://{s}.tile.openstreetmap.org/{z}/{x}/{y}.png', {
        attribution: '© OpenStreetMap contributors'
    }).addTo(mapa);
    
    <?php if (isset($_POST['ver_todas_mapa']) && !empty($todas_ubicaciones)): ?>
        // Mostrar todas las ubicaciones
        var bounds = [];
        <?php foreach ($todas_ubicaciones as $ubicacion): ?>
            var marker = L.marker([<?php echo $ubicacion['latitud']; ?>, <?php echo $ubicacion['longitud']; ?>])
                .bindPopup("<?php echo htmlspecialchars($ubicacion['nompa'] . ' ' . $ubicacion['apepa']); ?><br>Fecha: <?php echo date('H:i:s d/m/Y', strtotime($ubicacion['fecha_registro'])); ?>")
                .addTo(mapa);
            bounds.push([<?php echo $ubicacion['latitud']; ?>, <?php echo $ubicacion['longitud']; ?>]);
        <?php endforeach; ?>
        mapa.fitBounds(bounds);
    <?php else: ?>
        // Mostrar ubicación individual
        var marker = L.marker([<?php echo $latitud_busqueda; ?>, <?php echo $longitud_busqueda; ?>])
            .bindPopup("<?php echo htmlspecialchars($nombre_busqueda); ?>")
            .addTo(mapa);
        mapa.setView([<?php echo $latitud_busqueda; ?>, <?php echo $longitud_busqueda; ?>], 15);
    <?php endif; ?>
<?php endif; ?>

// =====================================================
// REGISTRO AUTOMÁTICO DE UBICACIÓN AL ESCANEAR QR
// =====================================================
<?php if ($registro_automatico): ?>
document.addEventListener('DOMContentLoaded', function() {
    const cedula = '<?php echo addslashes($cedula_qr); ?>';
    
    // Mostrar mensaje de carga
    const container = document.querySelector('.container');
    const loadingDiv = document.createElement('div');
    loadingDiv.className = 'alert alert-info text-center';
    loadingDiv.innerHTML = '<h4>⏳ Registrando ubicación automáticamente...</h4><p>Por favor, espere mientras obtenemos su ubicación.</p>';
    container.insertBefore(loadingDiv, container.firstChild);
    
    // Obtener ubicación automáticamente
    if (navigator.geolocation) {
        const options = {
            enableHighAccuracy: true,
            timeout: 15000,
            maximumAge: 0
        };
        
        navigator.geolocation.getCurrentPosition(
            function(position) {
                const latitud = position.coords.latitude;
                const longitud = position.coords.longitude;
                
                // Actualizar mensaje
                loadingDiv.innerHTML = '<h4>✅ Ubicación obtenida</h4><p>Registrando en el sistema...</p>';
                
                // Crear y enviar formulario automáticamente
                const form = document.createElement('form');
                form.method = 'POST';
                form.action = '';
                
                const inputCedula = document.createElement('input');
                inputCedula.type = 'hidden';
                inputCedula.name = 'cedula';
                inputCedula.value = cedula;
                form.appendChild(inputCedula);
                
                const inputLat = document.createElement('input');
                inputLat.type = 'hidden';
                inputLat.name = 'latitud';
                inputLat.value = latitud;
                form.appendChild(inputLat);
                
                const inputLng = document.createElement('input');
                inputLng.type = 'hidden';
                inputLng.name = 'longitud';
                inputLng.value = longitud;
                form.appendChild(inputLng);
                
                const inputAuto = document.createElement('input');
                inputAuto.type = 'hidden';
                inputAuto.name = 'auto_register';
                inputAuto.value = '1';
                form.appendChild(inputAuto);
                
                document.body.appendChild(form);
                form.submit();
            },
            function(error) {
                let errorMessage = "❌ Error al obtener ubicación: ";
                
                switch(error.code) {
                    case error.PERMISSION_DENIED:
                        errorMessage += "Permiso denegado. Por favor, habilite la ubicación en su dispositivo.";
                        break;
                    case error.POSITION_UNAVAILABLE:
                        errorMessage += "Información de ubicación no disponible.";
                        break;
                    case error.TIMEOUT:
                        errorMessage += "Se agotó el tiempo de espera.";
                        break;
                    default:
                        errorMessage += "Error desconocido.";
                }
                
                loadingDiv.className = 'alert alert-danger text-center';
                loadingDiv.innerHTML = '<h4>' + errorMessage + '</h4><p>Por favor, active el GPS y recargue la página.</p>';
            },
            options
        );
    } else {
        loadingDiv.className = 'alert alert-danger text-center';
        loadingDiv.innerHTML = '<h4>❌ Su navegador no soporta geolocalización</h4>';
    }
});
<?php endif; ?>

// Función para cerrar el mensaje de agradecimiento
function cerrarMensaje() {
    const mensaje = document.getElementById('mensajeAgradecimiento');
    const overlay = document.getElementById('mensajeOverlay');
    
    if (mensaje && overlay) {
        mensaje.style.animation = 'fadeOut 0.3s ease-out';
        overlay.style.animation = 'fadeOut 0.3s ease-out';
        
        setTimeout(function() {
            mensaje.remove();
            overlay.remove();
        }, 300);
    }
}

// NO auto-cerrar para que el usuario vea el botón de WhatsApp

// =====================================================
// FUNCIÓN PARA ENVIAR WHATSAPP AL HACER CLIC
// =====================================================
<?php if (isset($_GET['success']) && $enviar_whatsapp && !empty($nombre_busqueda) && !empty($latitud_busqueda)): ?>
function enviarWhatsApp() {
    // Construir el mensaje de WhatsApp (igual al del correo)
    const nombrePersona = "<?php echo addslashes($nombre_busqueda); ?>";
    const fechaHora = "<?php echo addslashes($fecha_registro); ?>";
    const latitud = "<?php echo $latitud_busqueda; ?>";
    const longitud = "<?php echo $longitud_busqueda; ?>";
    
    // Mensaje formateado para WhatsApp
    const mensajeWhatsApp = `📍 *Notificación de Ubicación*

*Hogar de los Abuelos*

Hola, le informamos que se ha registrado la ubicación de *${nombrePersona}*.

Compartimos esta información para su conocimiento y seguimiento.

*📍 Detalles de Ubicación:*
📅 Fecha y Hora: ${fechaHora}
🌍 Coordenadas GPS:
   • Latitud: ${latitud}
   • Longitud: ${longitud}

🗺️ *Ver ubicación en Google Maps:*
https://www.google.com/maps?q=${latitud},${longitud}

👍 Puede revisar la ubicación haciendo clic en el enlace de arriba.

---
Hogar de los Abuelos
Sistema de Registro de Ubicación`;
    
    // Codificar el mensaje para URL
    const mensajeCodificado = encodeURIComponent(mensajeWhatsApp);
    
    // Número de WhatsApp (sin + ni espacios)
    const numeroWhatsApp = "573027294894";
    
    // Crear el enlace de WhatsApp
    const enlaceWhatsApp = `https://wa.me/${numeroWhatsApp}?text=${mensajeCodificado}`;
    
    // Abrir WhatsApp
    window.open(enlaceWhatsApp, '_blank');
    
    // Cambiar el texto del botón para indicar que se envió
    const btn = document.getElementById('btnEnviarWhatsApp');
    btn.innerHTML = '<i class="bx bx-check" style="font-size: 24px; vertical-align: middle;"></i> ¡Enviado!';
    btn.style.background = '#128C7E';
    btn.disabled = true;
}
<?php endif; ?>
</script>

<!-- Footer -->
<footer style="background: white; padding: 25px 0; margin-top: 50px; box-shadow: 0 -4px 20px rgba(0,0,0,0.1); border-radius: 20px 20px 0 0;">
    <div class="container text-center">
        <div style="color: var(--dark-color); margin-bottom: 10px;">
            <i class='bx bx-heart' style="color: #dc3545; font-size: 1.5rem;"></i>
        </div>
        <p style="margin: 0; color: #6c757d; font-size: 0.95rem;">
            <strong style="color: var(--dark-color);">Hogar de los Abuelos</strong> - Sistema de Ubicación
        </p>
        <p style="margin: 5px 0 0 0; color: #adb5bd; font-size: 0.85rem;">
            © <?php echo date('Y'); ?> Todos los derechos reservados
        </p>
    </div>
</footer>

<script src="https://cdn.jsdelivr.net/npm/bootstrap@5.3.0/dist/js/bootstrap.bundle.min.js"></script>
</body>
</html>