<?php
session_start();
require_once 'conexion.php';

// Variables para filtros
$filtro_nombre = isset($_GET['nombre']) ? trim($_GET['nombre']) : '';
$filtro_fecha_inicio = isset($_GET['fecha_inicio']) ? $_GET['fecha_inicio'] : date('Y-m-d', strtotime('-7 days'));
$filtro_fecha_fin = isset($_GET['fecha_fin']) ? $_GET['fecha_fin'] : date('Y-m-d');
$filtro_hora_inicio = isset($_GET['hora_inicio']) ? $_GET['hora_inicio'] : '00:00';
$filtro_hora_fin = isset($_GET['hora_fin']) ? $_GET['hora_fin'] : '23:59';

// Paginación
$registros_por_pagina = 20;
$pagina_actual = isset($_GET['pagina']) ? max(1, intval($_GET['pagina'])) : 1;
$offset = ($pagina_actual - 1) * $registros_por_pagina;

// Construir la consulta SQL con filtros
$sql_base = "SELECT u.id, u.nompa, u.documento, u.correo, u.latitud, u.longitud, u.fecha_registro,
              p.apepa
              FROM ubicaciones_completas u
              LEFT JOIN patients p ON u.documento = p.numhs
              WHERE 1=1";

$params = [];

// Filtro por nombre
if (!empty($filtro_nombre)) {
    $sql_base .= " AND (u.nompa LIKE :nombre OR p.apepa LIKE :nombre OR u.documento LIKE :nombre)";
    $params['nombre'] = "%{$filtro_nombre}%";
}

// Filtro por rango de fechas y horas
$datetime_inicio = $filtro_fecha_inicio . ' ' . $filtro_hora_inicio . ':00';
$datetime_fin = $filtro_fecha_fin . ' ' . $filtro_hora_fin . ':59';
$sql_base .= " AND u.fecha_registro BETWEEN :fecha_inicio AND :fecha_fin";
$params['fecha_inicio'] = $datetime_inicio;
$params['fecha_fin'] = $datetime_fin;

// Contar total de registros
$sql_count = "SELECT COUNT(*) as total FROM (" . $sql_base . ") as subquery";
$stmt_count = $conexion->prepare($sql_count);
$stmt_count->execute($params);
$total_registros = $stmt_count->fetch(PDO::FETCH_ASSOC)['total'];
$total_paginas = ceil($total_registros / $registros_por_pagina);

// Obtener registros con paginación
$sql = $sql_base . " ORDER BY u.fecha_registro DESC LIMIT :limit OFFSET :offset";
$stmt = $conexion->prepare($sql);
foreach ($params as $key => $value) {
    $stmt->bindValue($key, $value);
}
$stmt->bindValue(':limit', $registros_por_pagina, PDO::PARAM_INT);
$stmt->bindValue(':offset', $offset, PDO::PARAM_INT);
$stmt->execute();
$registros = $stmt->fetchAll(PDO::FETCH_ASSOC);

// Estadísticas
$sql_stats = "SELECT 
              COUNT(*) as total,
              COUNT(DISTINCT documento) as usuarios_unicos,
              MIN(fecha_registro) as primer_registro,
              MAX(fecha_registro) as ultimo_registro
              FROM ubicaciones_completas";
$stmt_stats = $conexion->prepare($sql_stats);
$stmt_stats->execute();
$estadisticas = $stmt_stats->fetch(PDO::FETCH_ASSOC);
?>

<!DOCTYPE html>
<html lang="es">
<head>
    <meta charset="UTF-8">
    <title>Historial de Ubicaciones</title>
    <link href="https://cdn.jsdelivr.net/npm/bootstrap@5.3.0/dist/css/bootstrap.min.css" rel="stylesheet">
    <link href="https://cdn.jsdelivr.net/npm/bootstrap-icons@1.10.0/font/bootstrap-icons.css" rel="stylesheet">
    <style>
        .sidebar {
            height: 100%;
            width: 240px;
            position: fixed;
            top: 0;
            left: 0;
            background-color: rgb(21, 73, 124);
            color: white;
            padding-top: 20px;
            z-index: 1000;
        }
        .sidebar a {
            color: white;
            padding: 10px 15px;
            text-decoration: none;
            display: block;
            transition: background-color 0.3s;
        }
        .sidebar a:hover, .sidebar a.active {
            background-color: #575757;
        }
        .sidebar h3 {
            padding: 10px 15px;
            margin: 0 0 20px 0;
        }
        .content {
            margin-left: 240px;
            padding: 20px;
        }
        .filter-card {
            background: white;
            border-radius: 10px;
            padding: 20px;
            box-shadow: 0 2px 10px rgba(0,0,0,0.1);
            margin-bottom: 20px;
        }
        .stats-card {
            background: linear-gradient(135deg, #667eea 0%, #764ba2 100%);
            color: white;
            border-radius: 10px;
            padding: 20px;
            margin-bottom: 20px;
        }
        .stats-card h5 {
            margin: 0;
            font-size: 1rem;
            opacity: 0.9;
        }
        .stats-card h3 {
            margin: 10px 0 0 0;
            font-size: 2rem;
            font-weight: bold;
        }
        .table-container {
            background: white;
            border-radius: 10px;
            padding: 20px;
            box-shadow: 0 2px 10px rgba(0,0,0,0.1);
        }
        .table thead {
            background-color: rgb(21, 73, 124);
            color: white;
        }
        .btn-view {
            padding: 5px 10px;
            font-size: 0.875rem;
        }
        .pagination {
            justify-content: center;
            margin-top: 20px;
        }
        .badge-custom {
            padding: 5px 10px;
            border-radius: 5px;
        }
        .no-records {
            text-align: center;
            padding: 40px;
            color: #6c757d;
        }
        .no-records i {
            font-size: 4rem;
            margin-bottom: 20px;
        }
    </style>
</head>
<body class="bg-light">

<!-- Sidebar -->
<div class="sidebar">
    <h3 class="text-center text-white">Menú</h3>
    <a href="historial_ubicaciones.php" class="active"><i class="bi bi-clock-history"></i> Historial</a>
</div>

<!-- Content -->
<div class="content">
    <div class="container-fluid">
        <h1 class="mb-4"><i class="bi bi-clock-history"></i> Historial de Ubicaciones</h1>
        
        <!-- Estadísticas -->
        <div class="row mb-4">
            <div class="col-md-3">
                <div class="stats-card">
                    <h5>Total Registros</h5>
                    <h3><?php echo number_format($estadisticas['total']); ?></h3>
                </div>
            </div>
            <div class="col-md-3">
                <div class="stats-card">
                    <h5>Usuarios Únicos</h5>
                    <h3><?php echo number_format($estadisticas['usuarios_unicos']); ?></h3>
                </div>
            </div>
            <div class="col-md-3">
                <div class="stats-card">
                    <h5>Primer Registro</h5>
                    <h3><?php echo $estadisticas['primer_registro'] ? date('d/m/Y', strtotime($estadisticas['primer_registro'])) : 'N/A'; ?></h3>
                </div>
            </div>
            <div class="col-md-3">
                <div class="stats-card">
                    <h5>Último Registro</h5>
                    <h3><?php echo $estadisticas['ultimo_registro'] ? date('d/m/Y', strtotime($estadisticas['ultimo_registro'])) : 'N/A'; ?></h3>
                </div>
            </div>
        </div>

        <!-- Filtros -->
        <div class="filter-card">
            <h4 class="mb-3"><i class="bi bi-funnel"></i> Filtros de Búsqueda</h4>
            <form method="GET" action="">
                <div class="row g-3">
                    <div class="col-md-4">
                        <label for="nombre" class="form-label">Buscar por Nombre o Documento</label>
                        <input type="text" class="form-control" id="nombre" name="nombre" 
                               value="<?php echo htmlspecialchars($filtro_nombre); ?>" 
                               placeholder="Ej: Juan, Pérez, 123456">
                    </div>
                    <div class="col-md-2">
                        <label for="fecha_inicio" class="form-label">Fecha Inicio</label>
                        <input type="date" class="form-control" id="fecha_inicio" name="fecha_inicio" 
                               value="<?php echo $filtro_fecha_inicio; ?>">
                    </div>
                    <div class="col-md-2">
                        <label for="hora_inicio" class="form-label">Hora Inicio</label>
                        <input type="time" class="form-control" id="hora_inicio" name="hora_inicio" 
                               value="<?php echo $filtro_hora_inicio; ?>">
                    </div>
                    <div class="col-md-2">
                        <label for="fecha_fin" class="form-label">Fecha Fin</label>
                        <input type="date" class="form-control" id="fecha_fin" name="fecha_fin" 
                               value="<?php echo $filtro_fecha_fin; ?>">
                    </div>
                    <div class="col-md-2">
                        <label for="hora_fin" class="form-label">Hora Fin</label>
                        <input type="time" class="form-control" id="hora_fin" name="hora_fin" 
                               value="<?php echo $filtro_hora_fin; ?>">
                    </div>
                </div>
                <div class="row mt-3">
                    <div class="col-md-12">
                        <button type="submit" class="btn btn-primary">
                            <i class="bi bi-search"></i> Buscar
                        </button>
                        <a href="historial_ubicaciones.php" class="btn btn-secondary">
                            <i class="bi bi-arrow-counterclockwise"></i> Limpiar Filtros
                        </a>
                        <button type="button" class="btn btn-success" onclick="exportarExcel()">
                            <i class="bi bi-file-earmark-excel"></i> Exportar Excel
                        </button>
                    </div>
                </div>
            </form>
        </div>

        <!-- Información de resultados -->
        <div class="alert alert-info">
            <i class="bi bi-info-circle"></i> 
            Mostrando <strong><?php echo count($registros); ?></strong> de <strong><?php echo $total_registros; ?></strong> registros
            (Página <?php echo $pagina_actual; ?> de <?php echo max(1, $total_paginas); ?>)
        </div>

        <!-- Tabla de registros -->
        <div class="table-container">
            <?php if (count($registros) > 0): ?>
                <div class="table-responsive">
                    <table class="table table-hover table-striped">
                        <thead>
                            <tr>
                                <th>ID</th>
                                <th>Fecha y Hora</th>
                                <th>Nombre</th>
                                <th>Apellido</th>
                                <th>Documento</th>
                                <th>Correo</th>
                                <th>Coordenadas</th>
                                <th>Acciones</th>
                            </tr>
                        </thead>
                        <tbody>
                            <?php foreach ($registros as $registro): ?>
                                <tr>
                                    <td><?php echo $registro['id']; ?></td>
                                    <td>
                                        <strong><?php echo date('d/m/Y', strtotime($registro['fecha_registro'])); ?></strong><br>
                                        <small class="text-muted"><?php echo date('H:i:s', strtotime($registro['fecha_registro'])); ?></small>
                                    </td>
                                    <td><?php echo htmlspecialchars($registro['nompa']); ?></td>
                                    <td><?php echo htmlspecialchars($registro['apepa'] ?? 'N/A'); ?></td>
                                    <td><span class="badge badge-custom bg-secondary"><?php echo htmlspecialchars($registro['documento']); ?></span></td>
                                    <td><?php echo htmlspecialchars($registro['correo']); ?></td>
                                    <td>
                                        <small>
                                            Lat: <?php echo number_format($registro['latitud'], 6); ?><br>
                                            Lng: <?php echo number_format($registro['longitud'], 6); ?>
                                        </small>
                                    </td>
                                    <td>
                                        <a href="https://www.google.com/maps?q=<?php echo $registro['latitud']; ?>,<?php echo $registro['longitud']; ?>" 
                                           target="_blank" class="btn btn-sm btn-primary btn-view" title="Ver en Google Maps">
                                            <i class="bi bi-geo-alt"></i> Ver Mapa
                                        </a>
                                    </td>
                                </tr>
                            <?php endforeach; ?>
                        </tbody>
                    </table>
                </div>

                <!-- Paginación -->
                <?php if ($total_paginas > 1): ?>
                    <nav aria-label="Paginación">
                        <ul class="pagination">
                            <?php if ($pagina_actual > 1): ?>
                                <li class="page-item">
                                    <a class="page-link" href="?<?php echo http_build_query(array_merge($_GET, ['pagina' => $pagina_actual - 1])); ?>">
                                        Anterior
                                    </a>
                                </li>
                            <?php endif; ?>

                            <?php for ($i = max(1, $pagina_actual - 2); $i <= min($total_paginas, $pagina_actual + 2); $i++): ?>
                                <li class="page-item <?php echo $i == $pagina_actual ? 'active' : ''; ?>">
                                    <a class="page-link" href="?<?php echo http_build_query(array_merge($_GET, ['pagina' => $i])); ?>">
                                        <?php echo $i; ?>
                                    </a>
                                </li>
                            <?php endfor; ?>

                            <?php if ($pagina_actual < $total_paginas): ?>
                                <li class="page-item">
                                    <a class="page-link" href="?<?php echo http_build_query(array_merge($_GET, ['pagina' => $pagina_actual + 1])); ?>">
                                        Siguiente
                                    </a>
                                </li>
                            <?php endif; ?>
                        </ul>
                    </nav>
                <?php endif; ?>
            <?php else: ?>
                <div class="no-records">
                    <i class="bi bi-inbox text-muted"></i>
                    <h4>No se encontraron registros</h4>
                    <p class="text-muted">No hay ubicaciones registradas con los filtros seleccionados.</p>
                </div>
            <?php endif; ?>
        </div>
    </div>
</div>

<script src="https://cdn.jsdelivr.net/npm/bootstrap@5.3.0/dist/js/bootstrap.bundle.min.js"></script>
<script>
function exportarExcel() {
    // Obtener los parámetros actuales de la URL
    const params = new URLSearchParams(window.location.search);
    params.set('export', 'excel');
    
    // Crear el CSV
    let csv = 'ID,Fecha,Hora,Nombre,Apellido,Documento,Correo,Latitud,Longitud\n';
    
    <?php foreach ($registros as $registro): ?>
    csv += '<?php echo $registro['id']; ?>,' +
           '<?php echo date('d/m/Y', strtotime($registro['fecha_registro'])); ?>,' +
           '<?php echo date('H:i:s', strtotime($registro['fecha_registro'])); ?>,' +
           '"<?php echo str_replace('"', '""', $registro['nompa']); ?>",' +
           '"<?php echo str_replace('"', '""', $registro['apepa'] ?? ''); ?>",' +
           '<?php echo $registro['documento']; ?>,' +
           '"<?php echo str_replace('"', '""', $registro['correo']); ?>",' +
           '<?php echo $registro['latitud']; ?>,' +
           '<?php echo $registro['longitud']; ?>\n';
    <?php endforeach; ?>
    
    // Crear el blob y descargarlo
    const blob = new Blob([csv], { type: 'text/csv;charset=utf-8;' });
    const link = document.createElement('a');
    const url = URL.createObjectURL(blob);
    link.setAttribute('href', url);
    link.setAttribute('download', 'historial_ubicaciones_' + new Date().toISOString().slice(0,10) + '.csv');
    link.style.visibility = 'hidden';
    document.body.appendChild(link);
    link.click();
    document.body.removeChild(link);
}
</script>
</body>
</html>
