<?php
session_start();
$mensaje = "";
$ubicacion_guardada = false;

// Incluir el archivo de conexión
require_once 'conexion.php';
// Incluir configuración de email
require_once 'email_config.php';

if ($_SERVER["REQUEST_METHOD"] == "POST") {
    $qr_input = trim($_POST['qr_input'] ?? '');
    $ubicacion = trim($_POST['ubicacion'] ?? '');

    if ($qr_input !== '' && $ubicacion !== '') {
        $coordenadas = explode(",", $ubicacion);
        if (count($coordenadas) == 2) {
            $latitud = floatval(trim($coordenadas[0]));
            $longitud = floatval(trim($coordenadas[1]));

            if ($latitud >= -90 && $latitud <= 90 && $longitud >= -180 && $longitud <= 180) {
                // Buscar paciente por código QR
                $sql = "SELECT numhs, nompa, apepa, correo FROM patients WHERE qrInput = :qr";
                $stmt = $conexion->prepare($sql);
                $stmt->execute(['qr' => $qr_input]);

                if ($paciente = $stmt->fetch(PDO::FETCH_ASSOC)) {
                    $_SESSION['documento_paciente'] = $paciente['numhs'];
                    $_SESSION['nombre_paciente'] = $paciente['nompa'];
                    $_SESSION['apellido_paciente'] = $paciente['apepa'];
                    $_SESSION['correo_paciente'] = $paciente['correo'];

                    // Insertar ubicación
                    $stmt = $conexion->prepare("INSERT INTO ubicaciones_completas (nompa, documento, correo, latitud, longitud) VALUES (:nompa, :documento, :correo, :latitud, :longitud)");
                    
                    if ($stmt->execute([
                        'nompa' => $paciente['nompa'],
                        'documento' => $paciente['numhs'],
                        'correo' => $paciente['correo'],
                        'latitud' => $latitud,
                        'longitud' => $longitud
                    ])) {
                        // Enviar correo de alerta
                        $nombre_completo = $paciente['nompa'] . ' ' . $paciente['apepa'];
                        $fecha_hora = date('d/m/Y H:i:s');
                        
                        // Correo de destino fijo para alertas
                        $correo_destino = 'adultomayor@sibate-cundinamarca.gov.co';
                        
                        // Enviar correo al destinatario fijo
                        enviarCorreoUbicacion(
                            $correo_destino,
                            $nombre_completo,
                            $latitud,
                            $longitud,
                            $fecha_hora
                        );
                        
                        $mensaje = "✅ Ubicación registrada correctamente para: {$paciente['nompa']} {$paciente['apepa']}.";
                        $ubicacion_guardada = true;
                    } else {
                        $mensaje = "❌ Error al registrar ubicación.";
                    }
                } else {
                    $mensaje = "⚠️ No se encontró Usuario con ese código QR.";
                }
            } else {
                $mensaje = "❌ Coordenadas fuera de rango válido.";
            }
        } else {
            $mensaje = "❌ Formato de coordenadas inválido. Usa el botón 'Obtener ubicación'.";
        }
    } else {
        $mensaje = "❌ Todos los campos son obligatorios.";
    }
}
?>

<!DOCTYPE html>
<html lang="es">
<head>
    <meta charset="UTF-8">
    <title>Registro de Ubicación QR</title>
    <link href="https://cdn.jsdelivr.net/npm/bootstrap@5.3.0/dist/css/bootstrap.min.css" rel="stylesheet">
    <style>
        .sidebar {
            height: 100%;
            width: 240px;
            position: fixed;
            top: 0;
            left: 0;
            background-color: rgb(21, 73, 124);
            color: white;
            padding-top: 20px;
            z-index: 1000;
        }
        .sidebar a {
            color: white;
            padding: 10px 15px;
            text-decoration: none;
            display: block;
        }
        .sidebar a:hover {
            background-color: #575757;
        }
        .content {
            margin-left: 240px;
            padding: 20px;
        }
        .debug-info {
            margin-top: 15px;
            padding: 10px;
            background-color: #f8f9fa;
            border: 1px solid #ddd;
            border-radius: 5px;
            font-family: monospace;
        }
        .location-status {
            padding: 10px;
            margin: 10px 0;
            border-radius: 5px;
        }
        .warning-banner {
            padding: 10px;
            background-color: #fff3cd;
            border: 1px solid #ffeeba;
            color: #856404;
            border-radius: 5px;
            margin-bottom: 15px;
            font-size: 14px;
        }
        #qr-video {
            width: 100%;
            max-width: 640px;
            margin: 0 auto;
        }
        #qr-canvas {
            display: none;
        }
        .qr-overlay {
            position: fixed;
            top: 0;
            left: 0;
            right: 0;
            bottom: 0;
            background: rgba(0,0,0,0.9);
            display: none;
            justify-content: center;
            align-items: center;
            z-index: 9999;
        }
        .qr-container {
            position: relative;
            width: 100%;
            max-width: 640px;
            margin: 20px;
        }
        .close-qr {
            position: absolute;
            top: -40px;
            right: 0;
            color: white;
            font-size: 24px;
            cursor: pointer;
            background: none;
            border: none;
        }
        .switch-camera {
            position: absolute;
            top: -40px;
            left: 0;
            color: white;
            font-size: 18px;
            cursor: pointer;
            background: rgba(255,255,255,0.2);
            border: 2px solid white;
            border-radius: 5px;
            padding: 5px 15px;
            transition: all 0.3s;
        }
        .switch-camera:hover {
            background: rgba(255,255,255,0.3);
        }
        @media (max-width: 768px) {
            .sidebar {
                width: 100%;
                height: auto;
                position: relative;
            }
            .content {
                margin-left: 0;
                padding: 10px;
            }
            .container {
                padding: 0;
            }
            h1 {
                font-size: 1.5rem;
            }
            .form-label {
                font-size: 14px;
            }
            .btn {
                font-size: 14px;
                padding: 10px;
            }
            .switch-camera {
                font-size: 14px;
                padding: 5px 10px;
                top: -35px;
            }
            .close-qr {
                top: -35px;
                font-size: 20px;
            }
            .qr-container {
                margin: 10px;
            }
            .warning-banner {
                font-size: 12px;
            }
        }
        
        @media (max-width: 480px) {
            .content {
                padding: 5px;
            }
            h1 {
                font-size: 1.2rem;
                margin-bottom: 15px;
            }
            .p-4 {
                padding: 15px !important;
            }
            .btn {
                font-size: 13px;
                padding: 8px;
            }
            input[type="text"] {
                font-size: 14px;
            }
            .switch-camera {
                font-size: 12px;
                padding: 4px 8px;
            }
        }
    </style>
</head>
<body class="bg-light">

<!-- Sidebar -->
<div class="sidebar">
    <h3 class="text-center text-white">Menú</h3>
    <a href="registro_ubicacion.php">📍 Registro de Ubicación</a>
    <a href="ubicar_abuelos.php">🗺️ Ubicar Personas</a>
    <a href="historial_ubicaciones.php">📋 Historial</a>
</div>

<!-- QR Scanner Overlay -->
<div id="qr-overlay" class="qr-overlay">
    <div class="qr-container">
        <button class="switch-camera" onclick="cambiarCamara()" title="Cambiar cámara">🔄 Cambiar Cámara</button>
        <button class="close-qr" onclick="cerrarEscaneoQR()">×</button>
        <video id="qr-video" playsinline></video>
        <canvas id="qr-canvas"></canvas>
    </div>
</div>

<!-- Contenido principal -->
<div class="content">
    <div class="container mt-5">
        <h1 class="text-center mb-4">Registro de Ubicación por Código QR</h1>

        <?php if (!empty($mensaje)): ?>
            <div class="alert <?php echo $ubicacion_guardada ? 'alert-success' : 'alert-danger'; ?> text-center">
                <?php echo $mensaje; ?>
            </div>
        <?php endif; ?>

        <form method="POST" class="p-4 bg-white border rounded shadow-sm">
            <div class="mb-3">
                <label for="qr_input" class="form-label">Cédula (o QR):</label>
                <input type="text" name="qr_input" id="qr_input" class="form-control" required>
                <button type="button" class="btn btn-primary mt-2" onclick="iniciarEscaneo()">Escanear QR</button>
            </div>
            <div class="mb-3">
                <label for="ubicacion" class="form-label">Ubicación actual (Latitud,Longitud):</label>
                <div class="input-group">
                    <input type="text" name="ubicacion" id="ubicacion" class="form-control" readonly required>
                    <button type="button" class="btn btn-secondary" onclick="obtenerUbicacion()">Obtener ubicación</button>
                </div>
                <div class="warning-banner mt-2">
                    <strong>Importante:</strong> Para mejorar la precisión, asegúrese de estar en un lugar abierto y mantener el dispositivo quieto durante la captura.
                </div>
            </div>

            <div id="locationStatus" class="location-status"></div>
            
            <!-- Información de precisión -->
            <div class="debug-info" id="debugInfo" style="display: none;">
                <h5>Información detallada:</h5>
                <div id="debugContent"></div>
            </div>
            
            <button type="submit" class="btn btn-success w-100 mt-3">Registrar Ubicación</button>
        </form>
    </div>
</div>

<!-- Scripts -->
<script src="https://unpkg.com/jsqr@1.4.0/dist/jsQR.js"></script>
<script src="https://cdn.jsdelivr.net/npm/sweetalert2@11"></script>

<script>
// Función mejorada para obtener la ubicación con máxima precisión
function obtenerUbicacion() {
    const statusElement = document.getElementById('locationStatus');
    const debugElement = document.getElementById('debugInfo');
    const debugContent = document.getElementById('debugContent');
    const ubicacionInput = document.getElementById('ubicacion');
    
    statusElement.innerHTML = '<div class="alert alert-info">⏳ Obteniendo ubicación con alta precisión, espere por favor...</div>';
    debugElement.style.display = 'block';
    debugContent.innerHTML = '⚙️ Iniciando sistema de geolocalización de alta precisión...';
    
    if (navigator.geolocation) {
        const options = {
            enableHighAccuracy: true,
            timeout: 30000,
            maximumAge: 0
        };
        
        let waitingMessage = setTimeout(() => {
            statusElement.innerHTML = '<div class="alert alert-warning">⏳ Aún obteniendo la mejor señal posible. Este proceso puede tomar hasta 30 segundos para mayor precisión...</div>';
        }, 5000);
        
        navigator.geolocation.getCurrentPosition(
            function(position) {
                clearTimeout(waitingMessage);
                
                const lat = position.coords.latitude;
                const lng = position.coords.longitude;
                const accuracy = position.coords.accuracy;
                const timestamp = new Date(position.timestamp).toLocaleString();
                
                const formattedLat = lat.toFixed(8);
                const formattedLng = lng.toFixed(8);
                
                ubicacionInput.value = formattedLat + "," + formattedLng;
                
                let precisionClass = 'danger';
                let precisionText = 'Baja';
                
                if (accuracy < 10) {
                    precisionClass = 'success';
                    precisionText = 'Excelente';
                } else if (accuracy < 30) {
                    precisionClass = 'primary';
                    precisionText = 'Buena';
                } else if (accuracy < 100) {
                    precisionClass = 'warning';
                    precisionText = 'Regular';
                }
                
                debugContent.innerHTML = `
                    <b>Latitud:</b> ${formattedLat} (8 decimales ≈ precisión de centímetros)<br>
                    <b>Longitud:</b> ${formattedLng} (8 decimales ≈ precisión de centímetros)<br>
                    <b>Precisión:</b> ±${accuracy.toFixed(2)} metros<br>
                    <b>Calidad de señal:</b> ${precisionText}<br>
                    <b>Hora de captura:</b> ${timestamp}
                `;
                
                statusElement.innerHTML = `
                    <div class="alert alert-${precisionClass}">
                        ✅ Ubicación obtenida con precisión ${precisionText.toLowerCase()} (±${accuracy.toFixed(2)}m)
                    </div>`;
            },
            function(error) {
                clearTimeout(waitingMessage);
                let errorMessage = "❌ Error al obtener ubicación: ";
                
                switch(error.code) {
                    case error.PERMISSION_DENIED:
                        errorMessage += "Permiso denegado. Por favor, habilite la ubicación en su dispositivo.";
                        break;
                    case error.POSITION_UNAVAILABLE:
                        errorMessage += "Información de ubicación no disponible. Intente en un área más abierta.";
                        break;
                    case error.TIMEOUT:
                        errorMessage += "Se agotó el tiempo de espera. Intente nuevamente.";
                        break;
                    default:
                        errorMessage += "Error desconocido.";
                }
                
                statusElement.innerHTML = `<div class="alert alert-danger">${errorMessage}</div>`;
                debugElement.style.display = 'none';
            },
            options
        );
    } else {
        statusElement.innerHTML = '<div class="alert alert-danger">❌ Su navegador no soporta geolocalización</div>';
        debugElement.style.display = 'none';
    }
}

let videoStream = null;
let currentFacingMode = 'environment'; // Cámara posterior por defecto
let isScanning = false;

// Función para iniciar el escaneo de QR
function iniciarEscaneo() {
    const qrOverlay = document.getElementById('qr-overlay');
    const video = document.getElementById('qr-video');
    const canvas = document.getElementById('qr-canvas');
    const qrInput = document.getElementById('qr_input');

    // Mostrar el overlay
    qrOverlay.style.display = 'flex';

    iniciarCamara();
}

// Función para iniciar la cámara con el facing mode actual
function iniciarCamara() {
    const video = document.getElementById('qr-video');
    const canvas = document.getElementById('qr-canvas');
    const qrInput = document.getElementById('qr_input');

    // Configurar el video
    navigator.mediaDevices.getUserMedia({ 
        video: { 
            facingMode: currentFacingMode,
            width: { ideal: 1280 },
            height: { ideal: 720 }
        } 
    }).then(stream => {
        videoStream = stream;
        video.srcObject = stream;
        video.play();
        isScanning = true;

        // Función para escanear frames
        function escanearFrame() {
            if (!isScanning) return;
            if (video.readyState === video.HAVE_ENOUGH_DATA) {
                const canvasElement = canvas.getContext('2d');
                canvas.height = video.videoHeight;
                canvas.width = video.videoWidth;
                canvasElement.drawImage(video, 0, 0, canvas.width, canvas.height);
                
                const imageData = canvasElement.getImageData(0, 0, canvas.width, canvas.height);
                const code = jsQR(imageData.data, imageData.width, imageData.height);

                if (code) {
                    // QR detectado
                    qrInput.value = code.data;
                    cerrarEscaneoQR();
                    Swal.fire({
                        icon: 'success',
                        title: '¡Código QR detectado!',
                        text: `Valor escaneado: ${code.data}`,
                        timer: 2000,
                        showConfirmButton: false
                    });
                    return;
                }
            }
            requestAnimationFrame(escanearFrame);
        }

        requestAnimationFrame(escanearFrame);
    }).catch(error => {
        console.error('Error accediendo a la cámara:', error);
        Swal.fire({
            icon: 'error',
            title: 'Error de cámara',
            text: 'No se pudo acceder a la cámara. Por favor, verifique los permisos.',
            confirmButtonText: 'Aceptar'
        });
        cerrarEscaneoQR();
    });
}

// Función para cambiar entre cámara frontal y posterior
function cambiarCamara() {
    // Alternar entre 'user' (frontal) y 'environment' (posterior)
    currentFacingMode = currentFacingMode === 'environment' ? 'user' : 'environment';
    
    // Detener el stream actual
    if (videoStream) {
        videoStream.getTracks().forEach(track => track.stop());
        videoStream = null;
    }
    
    // Reiniciar con la nueva cámara
    iniciarCamara();
}

// Función para cerrar el escáner QR
function cerrarEscaneoQR() {
    const qrOverlay = document.getElementById('qr-overlay');
    qrOverlay.style.display = 'none';
    isScanning = false;
    
    if (videoStream) {
        videoStream.getTracks().forEach(track => track.stop());
        videoStream = null;
    }
    
    // Reset a cámara posterior por defecto
    currentFacingMode = 'environment';
}

// Cerrar el escáner si se presiona la tecla Escape
document.addEventListener('keydown', function(event) {
    if (event.key === 'Escape') {
        cerrarEscaneoQR();
    }
});
</script>

</body>
</html>