<?php
/**
 * Sistema de Control de Acceso por Roles
 * Programa Persona Mayor
 */

// Iniciar sesión si no está iniciada
if (session_status() === PHP_SESSION_NONE) {
    session_start();
}

/**
 * Verifica si el usuario tiene acceso basado en su rol
 * @param array $allowedRoles Array de roles permitidos
 * @return bool
 */
function checkAccess($allowedRoles = []) {
    // Verificar si hay sesión activa
    if(!isset($_SESSION['rol']) || !isset($_SESSION['id'])) {
        header('location: ../login.php');
        exit();
    }
    
    // Si no se especifican roles, permitir acceso a cualquier usuario autenticado
    if(empty($allowedRoles)) {
        return true;
    }
    
    // Admin siempre tiene acceso
    if($_SESSION['rol'] == 1) {
        return true;
    }
    
    // Verificar si el rol del usuario está en los roles permitidos
    // O si tiene permisos personalizados configurados (más flexible)
    if(in_array($_SESSION['rol'], $allowedRoles)) {
        return true;
    }
    
    // Si tiene permisos personalizados, permitir acceso aunque el rol no coincida
    if(isset($_SESSION['has_custom_permissions']) && $_SESSION['has_custom_permissions'] === true) {
        return true;
    }
    
    // Si no cumple ninguna condición, denegar acceso
    header('location: ../acceso_denegado.php');
    exit();
}

/**
 * Obtiene el nombre del rol
 * @param int $rolId ID del rol
 * @return string Nombre del rol
 */
function getRoleName($rolId) {
    $roles = [
        1 => 'Administrador',
        3 => 'Profesor',
        4 => 'Psicólogo',
        5 => 'Fisioterapeuta',
        6 => 'Enfermería'
    ];
    
    return isset($roles[$rolId]) ? $roles[$rolId] : 'Usuario';
}

/**
 * Verifica si el usuario es administrador
 * @return bool
 */
function isAdmin() {
    return isset($_SESSION['rol']) && $_SESSION['rol'] == 1;
}

/**
 * Verifica si el usuario es profesional (roles 3-6)
 * @return bool
 */
function isProfessional() {
    return isset($_SESSION['rol']) && in_array($_SESSION['rol'], [3, 4, 5, 6]);
}

/**
 * Obtiene el dashboard correspondiente al rol
 * @param int $rolId ID del rol
 * @return string Ruta al dashboard
 */
function getDashboardByRole($rolId) {
    $dashboards = [
        1 => 'admin/escritorio.php',
        3 => 'profesionales/dashboard_profesor.php',
        4 => 'profesionales/dashboard_psicologo.php',
        5 => 'profesionales/dashboard_fisioterapeuta.php',
        6 => 'profesionales/dashboard_enfermeria.php'
    ];
    
    return isset($dashboards[$rolId]) ? $dashboards[$rolId] : 'login.php';
}

/**
 * Obtiene los permisos del módulo según el rol
 * @param int $rolId ID del rol
 * @return array Permisos del rol
 */
function getModulePermissions($rolId) {
    $permissions = [
        1 => [ // Administrador - Acceso total
            'dashboard' => true,
            'remisiones' => true,
            'usuarios' => true,
            'cursos' => true,
            'asistencia' => true,
            'profesionales' => true,
            'recursos' => true,
            'ajustes' => true,
            'acerca' => true,
            'reportes' => true
        ],
        3 => [ // Profesor
            'dashboard' => true,
            'remisiones' => false,
            'usuarios' => true,
            'cursos' => true,
            'asistencia' => true,
            'profesionales' => false,
            'recursos' => false,
            'ajustes' => false,
            'acerca' => true,
            'reportes' => false
        ],
        4 => [ // Psicólogo
            'dashboard' => true,
            'remisiones' => true,
            'usuarios' => true,
            'cursos' => false,
            'asistencia' => false,
            'profesionales' => false,
            'recursos' => false,
            'ajustes' => false,
            'acerca' => true,
            'reportes' => false
        ],
        5 => [ // Fisioterapeuta
            'dashboard' => true,
            'remisiones' => true,
            'usuarios' => true,
            'cursos' => false,
            'asistencia' => false,
            'profesionales' => false,
            'recursos' => false,
            'ajustes' => false,
            'acerca' => true,
            'reportes' => false
        ],
        6 => [ // Enfermería
            'dashboard' => true,
            'remisiones' => false,
            'usuarios' => true,
            'cursos' => false,
            'asistencia' => true,
            'profesionales' => false,
            'recursos' => false,
            'ajustes' => false,
            'acerca' => true,
            'reportes' => false
        ]
    ];
    
    return isset($permissions[$rolId]) ? $permissions[$rolId] : [];
}

/**
 * Verifica si el usuario tiene permiso para un módulo específico
 * @param string $module Nombre del módulo (sin prefijo perm_)
 * @return bool
 */
function hasModulePermission($module) {
    if(!isset($_SESSION['rol']) || !isset($_SESSION['id'])) {
        return false;
    }
    
    // Admin siempre tiene acceso total
    if($_SESSION['rol'] == 1) {
        return true;
    }
    
    // SOLO usar permisos personalizados (checkboxes)
    if(isset($_SESSION['has_custom_permissions']) && $_SESSION['has_custom_permissions'] === true) {
        // Usar permisos de la sesión (columnas sin prefijo perm_)
        if(isset($_SESSION['menu_permissions'][$module])) {
            return $_SESSION['menu_permissions'][$module] == 1;
        }
        // Si la columna no existe en permisos personalizados, denegar
        return false;
    }
    
    // Si no tiene permisos personalizados configurados, denegar acceso
    return false;
}

/**
 * Verifica si el usuario tiene permiso para un submenú específico
 * @param string $submenu Nombre del submenú (ej: 'remisiones_todas', 'usuarios_lista')
 * @return bool
 */
function hasSubmenuPermission($submenu) {
    if(!isset($_SESSION['rol']) || !isset($_SESSION['id'])) {
        return false;
    }
    
    // Admin siempre tiene acceso total
    if($_SESSION['rol'] == 1) {
        return true;
    }
    
    // SOLO usar permisos personalizados (checkboxes)
    if(isset($_SESSION['has_custom_permissions']) && $_SESSION['has_custom_permissions'] === true) {
        // Usar permisos de la sesión (ya cargados en el login)
        if(isset($_SESSION['menu_permissions'][$submenu])) {
            return $_SESSION['menu_permissions'][$submenu] == 1;
        }
        // Si la columna del submenú no existe en permisos, denegar
        return false;
    }
    
    // Si no tiene permisos personalizados configurados, denegar acceso
    return false;
}
?>
