<?php
// Incluir sistema de control de acceso
require '../../backend/php/check_access.php';

// Verificar permisos
if(!hasModulePermission('asistencia')){
    header('location: ../acceso_denegado.php');
    exit;
}

require '../../backend/bd/Conexion.php';

// Obtener datos del usuario logueado
$userId = $_SESSION['id'];
$stmt = $connect->prepare("SELECT username, name FROM users WHERE id = ?");
$stmt->execute([$userId]);
$userData = $stmt->fetch(PDO::FETCH_ASSOC);

$professionalName = $userData['name'] ?? $userData['username'] ?? 'Usuario';
$professionalPosition = 'Administrador del Sistema'; // Puedes personalizar esto según el rol
?>

<!DOCTYPE html>
<html lang="es">
<head>
    <meta charset="UTF-8">
    <title>Clínica Salud | Asistencia con QR</title>
    <link href='https://unpkg.com/boxicons@2.0.9/css/boxicons.min.css' rel='stylesheet'>
    <link rel="stylesheet" href="../../backend/css/admin.css?v=7.0">
    <link rel="stylesheet" href="https://cdn.datatables.net/1.11.5/css/jquery.dataTables.min.css">
    
    <style>
        .main-container {
            display: flex;
            gap: 30px;
            margin-top: 20px;
        }
        
        .qr-section {
            flex: 1;
            max-width: 350px;
        }
        
        .qr-container {
            background: white;
            border-radius: 10px;
            padding: 20px;
            box-shadow: 0 2px 10px rgba(0,0,0,0.05);
            margin-bottom: 20px;
        }
        
        #scanner {
            width: 100%;
            border-radius: 8px;
            border: 1px solid #ddd;
            margin: 15px 0;
            background: #f8f9fa;
        }
        
        .detected-container {
            background: #f8f9fa;
            border-radius: 8px;
            padding: 15px;
            margin-top: 15px;
            display: none;
            border: 1px solid #ddd;
        }
        
        .data-section {
            flex: 2;
        }
        
        .stats-grid {
            display: grid;
            grid-template-columns: repeat(2, 1fr);
            gap: 15px;
            margin-bottom: 20px;
        }
        
        .stat-card {
            background: white;
            border-radius: 8px;
            padding: 15px;
            box-shadow: 0 2px 10px rgba(0,0,0,0.05);
        }
        
        .stat-card h5 {
            color: #6c757d;
            font-size: 14px;
            margin-bottom: 5px;
        }
        
        .stat-card p {
            font-size: 22px;
            font-weight: 600;
            color: #1e2c4b;
            margin-bottom: 0;
        }
        
        .table-wrapper {
            background: white;
            border-radius: 10px;
            padding: 20px;
            box-shadow: 0 2px 10px rgba(0,0,0,0.05);
        }
        
        .badge {
            padding: 5px 10px;
            border-radius: 20px;
            font-size: 12px;
            font-weight: 600;
        }
        
        .badge-registered {
            background: #e6f7ee;
            color: #28a745;
        }
        
        .badge-success {
            background: #e6f7ee;
            color: #28a745;
        }
        
        .badge-warning {
            background: #fff3cd;
            color: #856404;
        }
        
        .section-title {
            display: flex;
            align-items: center;
            gap: 10px;
            color: #1e2c4b;
            margin-bottom: 15px;
        }
        
        .section-title i {
            font-size: 24px;
            color: #4968ad;
        }
        
        .camera-controls {
            display: flex;
            justify-content: center;
            margin-top: 10px;
            gap: 10px;
        }
        
        .switch-camera-btn {
            background: #4968ad;
            color: white;
            border: none;
            padding: 8px 15px;
            border-radius: 5px;
            cursor: pointer;
            font-size: 14px;
            transition: all 0.3s;
            display: flex;
            align-items: center;
            gap: 5px;
        }
        
        .switch-camera-btn:hover {
            background: #3a5491;
        }
        
        .switch-camera-btn i {
            font-size: 16px;
        }

        .btn {
            display: inline-block;
            font-weight: 400;
            text-align: center;
            white-space: nowrap;
            vertical-align: middle;
            user-select: none;
            border: 1px solid transparent;
            padding: 0.375rem 0.75rem;
            font-size: 1rem;
            line-height: 1.5;
            border-radius: 0.25rem;
            transition: all 0.15s ease-in-out;
            cursor: pointer;
        }
        
        .btn-primary {
            color: #fff;
            background-color: #4968ad;
            border-color: #4968ad;
        }
        
        .btn-primary:hover {
            background-color: #3c57a5;
            border-color: #3c57a5;
        }
        
        .btn i {
            margin-right: 5px;
        }
        
        .filter-section {
            background: white;
            border-radius: 12px;
            padding: 20px;
            margin-bottom: 20px;
            box-shadow: 0 4px 20px rgba(0, 0, 0, .08);
            border: 1px solid #f0f0f0;
        }
        
        .filter-header {
            display: flex;
            align-items: center;
            gap: 10px;
            margin-bottom: 15px;
            color: #2d3436;
        }
        
        .filter-header i {
            font-size: 22px;
            background: linear-gradient(135deg, #667eea 0%, #764ba2 100%);
            -webkit-background-clip: text;
            -webkit-text-fill-color: transparent;
            background-clip: text;
        }
        
        .filter-grid {
            display: grid;
            grid-template-columns: repeat(auto-fit, minmax(200px, 1fr));
            gap: 15px;
            align-items: end;
        }
        
        .filter-group {
            display: flex;
            flex-direction: column;
        }
        
        .filter-group label {
            font-size: 13px;
            font-weight: 600;
            color: #2d3436;
            margin-bottom: 6px;
        }
        
        .filter-group input,
        .filter-group select {
            padding: 10px 12px;
            border: 2px solid #e8e8e8;
            border-radius: 8px;
            font-size: 14px;
            transition: all 0.3s ease;
            background: #f8f9fc;
        }
        
        .filter-group input:focus,
        .filter-group select:focus {
            outline: none;
            border-color: #667eea;
            background: white;
            box-shadow: 0 0 0 3px rgba(102, 126, 234, 0.1);
        }
        
        .btn-filter {
            background: linear-gradient(135deg, #667eea 0%, #764ba2 100%);
            color: white;
            border: none;
            padding: 10px 20px;
            border-radius: 8px;
            cursor: pointer;
            font-weight: 600;
            transition: all 0.3s ease;
            display: inline-flex;
            align-items: center;
            gap: 8px;
        }
        
        .btn-filter:hover {
            transform: translateY(-2px);
            box-shadow: 0 6px 20px rgba(102, 126, 234, 0.4);
        }
        
        .btn-reset {
            background: #6c757d;
        }
        
        .btn-reset:hover {
            background: #5a6268;
        }
        
        /* RESPONSIVE DESIGN */
        @media (max-width: 992px) {
            .main-container {
                flex-direction: column;
                gap: 20px;
            }
            
            .qr-section {
                max-width: 100%;
            }
            
            .stats-grid {
                grid-template-columns: 1fr;
                gap: 10px;
            }
            
            .table-wrapper {
                overflow-x: auto;
            }
        }
        
        @media (max-width: 768px) {
            .main-container {
                margin-top: 10px;
            }
            
            .qr-container, .table-wrapper {
                padding: 15px;
            }
            
            #scanner {
                margin: 10px 0;
            }
            
            .section-title {
                font-size: 16px;
                flex-wrap: wrap;
            }
            
            .section-title h4 {
                font-size: 18px;
                margin: 0;
            }
            
            .section-title i {
                font-size: 20px;
            }
            
            .stat-card h5 {
                font-size: 12px;
            }
            
            .stat-card p {
                font-size: 18px;
            }
            
            .alert {
                padding: 12px 15px;
                font-size: 13px;
            }
            
            .alert i {
                font-size: 20px;
            }
            
            .btn {
                padding: 8px 12px;
                font-size: 14px;
            }
            
            .switch-camera-btn {
                padding: 6px 12px;
                font-size: 13px;
            }
            
            table {
                font-size: 12px;
            }
            
            .badge {
                font-size: 10px;
                padding: 3px 8px;
            }
        }
        
        @media (max-width: 576px) {
            h1.title {
                font-size: 24px;
            }
            
            .qr-container, .table-wrapper {
                padding: 12px;
                border-radius: 8px;
            }
            
            .section-title h4 {
                font-size: 16px;
            }
            
            .stats-grid {
                gap: 8px;
            }
            
            .stat-card {
                padding: 12px;
            }
            
            .stat-card h5 {
                font-size: 11px;
            }
            
            .stat-card p {
                font-size: 16px;
            }
            
            .btn {
                width: 100%;
                margin-bottom: 5px;
            }
            
            .camera-controls {
                margin-top: 8px;
            }
            
            .switch-camera-btn {
                width: 100%;
            }
            
            .detected-container {
                padding: 12px;
            }
            
            .alert {
                padding: 10px 12px;
                font-size: 12px;
            }
        }
        
        /* Estilos para los botones de registro */
        #opcionesContainer button:hover {
            transform: scale(1.02);
            box-shadow: 0 4px 12px rgba(0,0,0,0.15);
        }
        
        #opcionesContainer button:active {
            transform: scale(0.98);
        }
    </style>
</head>
<body>
      <!-- SIDEBAR -->
   <!-- SIDEBAR DINÁMICO -->
    <?php include '../../backend/php/sidebar_menu.php'; ?>
 <!-- SIDEBAR -->
    <!-- CONTENIDO PRINCIPAL -->
    <section id="content">
        <nav>
            <i class='bx bx-menu toggle-sidebar'></i>
            <form action="#">
                <div class="form-group">
                    <input type="text" placeholder="Buscar...">
                    <i class='bx bx-search icon'></i>
                </div>
            </form>
            <div class="profile">
                <img src="https://images.unsplash.com/photo-1517841905240-472988babdf9?ixid=MnwxMjA3fDB8MHxzZWFyY2h8NHx8cGVvcGxlfGVufDB8fDB8fA%3D%3D&ixlib=rb-1.2.1&auto=format&fit=crop&w=500&q=60" alt="">
                <ul class="profile-link">
                    <li><a href="../profile/mostrar.php"><i class='bx bxs-user-circle icon'></i> Perfil</a></li>
                    <li><a href="../salir.php"><i class='bx bxs-log-out-circle'></i> Salir</a></li>
                </ul>
            </div>
        </nav>

        <main>
            <h1 class="title">Asistencia con QR</h1>
            <ul class="breadcrumbs">
                <li><a href="../admin/escritorio.php">Home</a></li>
                <li class="divider">/</li>
                <li><a href="../cursos/registro_cursos.php">Cursos</a></li>
                <li class="divider">/</li>
                <li><a href="#" class="active">Asistencia QR</a></li>
            </ul>

            <?php
            if ($_SERVER['REQUEST_METHOD'] === 'POST' && !empty($_POST['qr_data'])) {
                $qrData = $_POST['qr_data'];
                $modo_registro = isset($_POST['modo_registro']) ? $_POST['modo_registro'] : 'NORMAL';
                $tipo = isset($_POST['tipo_asistencia']) ? $_POST['tipo_asistencia'] : 'ENTRADA';
                $motivo = isset($_POST['motivo_salida']) ? trim($_POST['motivo_salida']) : null;
                
                // Si es eventualidad, forzar tipo SALIDA
                if ($modo_registro === 'EVENTUALIDAD') {
                    $tipo = 'SALIDA';
                }
                
                // Extraer la cédula si el QR contiene una URL
                $documento = $qrData;
                if (strpos($qrData, 'cedula=') !== false) {
                    // Es una URL, extraer el parámetro cedula
                    parse_str(parse_url($qrData, PHP_URL_QUERY), $params);
                    $documento = isset($params['cedula']) ? $params['cedula'] : $qrData;
                }
                
                try {
                    // Verificar si el paciente existe en la base de datos
                    $stmt = $connect->prepare("SELECT numhs AS documento, nompa, apepa FROM patients WHERE numhs = ?");
                    $stmt->execute([$documento]);
                    $paciente = $stmt->fetch(PDO::FETCH_ASSOC);

                    if ($paciente) {
                        // Insertar con tipo y motivo
                        $insert = $connect->prepare("INSERT INTO asistencia_qr (documento, nompa, apepa, fecha, tipo, motivo) VALUES (?, ?, ?, NOW(), ?, ?)");
                        $insert->execute([$paciente['documento'], $paciente['nompa'], $paciente['apepa'], $tipo, $motivo]);
                        
                        $tipoTexto = $tipo == 'ENTRADA' ? 'Entrada' : 'Salida';
                        $mensaje = '<div class="alert alert-success"><strong>'.$tipoTexto.' registrada</strong> para '.$paciente['nompa'].' '.$paciente['apepa'];
                        if($tipo == 'SALIDA' && $motivo) {
                            $mensaje .= '<br><small>Motivo: '.$motivo.'</small>';
                        }
                        $mensaje .= '</div>';
                        echo $mensaje;
                    } else {
                        echo '<div class="alert alert-warning">Documento no registrado en el sistema: '.$documento.'</div>';
                    }
                } catch (PDOException $e) {
                    echo '<div class="alert alert-danger">Error: '.$e->getMessage().'</div>';
                }
            }
            ?>

            <div class="main-container">
                <!-- Sección QR -->
                <div class="qr-section">
                    <div class="qr-container">
                        <div class="section-title">
                            <i class='bx bx-qr-scan'></i>
                            <h4>Escanear código QR</h4>
                        </div>
                        <div id="scanner" style="width: 100%;"></div>
                        <div class="camera-controls">
                            <button type="button" class="switch-camera-btn" onclick="switchCamera()">
                                <i class='bx bx-camera'></i>
                                Cambiar Cámara
                            </button>
                        </div>
                        
                        <div class="detected-container">
                            <form method="POST" id="formAsistencia">
                                <h5 class="text-center mb-3">¡QR detectado!</h5>
                                <input type="hidden" id="detected-qr-code" name="qr_data">
                                <input type="hidden" id="tipo_asistencia" name="tipo_asistencia" value="">
                                <input type="hidden" id="modo_registro" name="modo_registro" value="">
                                <input type="hidden" id="motivo_salida" name="motivo_salida" value="">
                                
                                <!-- Selector directo: 3 opciones -->
                                <div id="opcionesContainer" style="margin-bottom: 15px;">
                                    <label style="display: block; font-weight: bold; margin-bottom: 12px; color: #333; text-align: center;">Seleccione una opción:</label>
                                    <div style="display: flex; gap: 10px; flex-direction: column;">
                                        <button type="button" onclick="registrarEntrada()" style="padding: 15px; background: #e6f7ee; color: #28a745; border: 2px solid #28a745; border-radius: 8px; font-weight: 600; font-size: 16px; cursor: pointer; transition: all 0.3s;">
                                            🟢 Entrada
                                        </button>
                                        <button type="button" onclick="registrarSalida()" style="padding: 15px; background: #fff3cd; color: #856404; border: 2px solid #ffc107; border-radius: 8px; font-weight: 600; font-size: 16px; cursor: pointer; transition: all 0.3s;">
                                            🔴 Salida
                                        </button>
                                        <button type="button" onclick="mostrarEventualidad()" style="padding: 15px; background: #f8d7da; color: #721c24; border: 2px solid #dc3545; border-radius: 8px; font-weight: 600; font-size: 16px; cursor: pointer; transition: all 0.3s;">
                                            ⚠️ Eventualidad
                                        </button>
                                    </div>
                                </div>
                                
                                <!-- Selector de tipo de eventualidad (oculto inicialmente) -->
                                <div id="eventualidadSelector" style="display: none; margin-bottom: 15px;">
                                    <label style="display: block; font-weight: bold; margin-bottom: 8px; color: #333; text-align: center;">Tipo de Eventualidad:</label>
                                    <select id="tipoEventualidad" onchange="registrarEventualidad()" style="width: 100%; padding: 12px; border: 2px solid #dc3545; border-radius: 8px; font-size: 15px; font-weight: 600; color: #721c24; background: #fff; cursor: pointer;">
                                        <option value="">-- Seleccione --</option>
                                        <option value="Enfermo">🤒 Enfermo</option>
                                        <option value="Urgencia">🚨 Urgencia</option>
                                    </select>
                                </div>
                                
                                <button type="button" class="btn btn-outline-secondary btn-block mt-2" onclick="resetScanner()" style="margin-top: 15px;">
                                    <i class='bx bx-reset'></i> Escanear otro
                                </button>
                            </form>
                        </div>
                    </div>
                </div>
                
                <!-- Sección de datos -->
                <div class="data-section">
                    <!-- Filtros de fecha -->
                    <div class="filter-section">
                        <div class="filter-header">
                            <i class='bx bx-filter-alt'></i>
                            <h4 style="margin: 0; font-size: 16px;">Filtros de Exportación</h4>
                        </div>
                        <div class="filter-grid">
                            <div class="filter-group">
                                <label for="filterType">Tipo de Filtro</label>
                                <select id="filterType" onchange="toggleDateInputs()">
                                    <option value="all">Todas las asistencias</option>
                                    <option value="today">Solo hoy</option>
                                    <option value="single">Fecha específica</option>
                                    <option value="range">Rango de fechas</option>
                                </select>
                            </div>
                            <div class="filter-group" id="singleDateGroup" style="display: none;">
                                <label for="singleDate">Fecha</label>
                                <input type="date" id="singleDate">
                            </div>
                            <div class="filter-group" id="startDateGroup" style="display: none;">
                                <label for="startDate">Fecha Inicio</label>
                                <input type="date" id="startDate">
                            </div>
                            <div class="filter-group" id="endDateGroup" style="display: none;">
                                <label for="endDate">Fecha Fin</label>
                                <input type="date" id="endDate">
                            </div>
                            <div class="filter-group">
                                <label for="filterCentro">Modalidad</label>
                                <select id="filterCentro">
                                    <option value="all">Todos</option>
                                    <?php
                                    try {
                                        $modalidades = $connect->query("SELECT DISTINCT modalidad FROM patients WHERE modalidad IS NOT NULL AND modalidad != '' ORDER BY modalidad");
                                        while($mod = $modalidades->fetch(PDO::FETCH_ASSOC)) {
                                            echo '<option value="'.htmlspecialchars($mod['modalidad']).'">'.htmlspecialchars($mod['modalidad']).'</option>';
                                        }
                                    } catch (PDOException $e) {
                                        echo '<option value="">Error al cargar centros</option>';
                                    }
                                    ?>
                                </select>
                            </div>
                            <div class="filter-group">
                                <button class="btn-filter" onclick="applyFilter()">
                                    <i class='bx bx-check-circle'></i>
                                    Aplicar Filtro
                                </button>
                            </div>
                        </div>
                        
                        <div class="filter-grid" style="margin-top: 15px; padding-top: 15px; border-top: 1px solid #e8e8e8;">
                            <div class="filter-group">
                                <label>
                                    <i class='bx bx-user'></i> Profesional que genera el reporte
                                </label>
                                <div style="padding: 10px 12px; background: #f8f9fc; border: 2px solid #e8e8e8; border-radius: 8px; color: #2d3436; font-weight: 600;">
                                    <?php echo htmlspecialchars($professionalName); ?>
                                </div>
                            </div>
                            <div class="filter-group">
                                <label>
                                    <i class='bx bx-briefcase'></i> Cargo
                                </label>
                                <div style="padding: 10px 12px; background: #f8f9fc; border: 2px solid #e8e8e8; border-radius: 8px; color: #2d3436;">
                                    <?php echo htmlspecialchars($professionalPosition); ?>
                                </div>
                            </div>
                        </div>
                        <input type="hidden" id="professionalName" value="<?php echo htmlspecialchars($professionalName); ?>">
                        <input type="hidden" id="professionalPosition" value="<?php echo htmlspecialchars($professionalPosition); ?>">
                    </div>
                    
                    <div class="stats-grid">
                        <div class="stat-card">
                            <h5>Total Asistencias</h5>
                            <?php
                            $total = $connect->query("SELECT COUNT(*) FROM asistencia_qr")->fetchColumn();
                            echo "<p>$total</p>";
                            ?>
                        </div>
                        <div class="stat-card">
                            <h5>Asistencias Hoy</h5>
                            <?php
                            $hoy = $connect->query("SELECT COUNT(*) FROM asistencia_qr WHERE DATE(fecha) = CURDATE()")->fetchColumn();
                            echo "<p>$hoy</p>";
                            ?>
                        </div>
                    </div>
                    
                    <div class="table-wrapper">
                        <div class="section-title">
                            <i class='bx bx-list-ul'></i>
                            <h4>Registro de Asistencias <span id="filterInfo" style="font-size: 12px; color: #6c757d; font-weight: normal;"></span></h4>
                            <div style="margin-left: auto; display: flex; gap: 10px; flex-wrap: wrap;">
                                <button id="downloadPdfBtn" class="btn btn-primary" style="background: linear-gradient(135deg, #667eea 0%, #764ba2 100%); border: none;">
                                    <i class='bx bxs-file-pdf'></i> Descargar PDF
                                </button>
                                <button id="downloadBtn" class="btn btn-primary">
                                    <i class='bx bx-download'></i> Descargar CSV
                                </button>
                            </div>
                        </div>
                        
                        <table id="attendanceTable" class="display" style="width:100%">
                            <thead>
                                <tr>
                                    <th>#</th>
                                    <th>Documento</th>
                                    <th>Nombre</th>
                                    <th>Apellido</th>
                                    <th>Edad</th>
                                    <th>Modalidad</th>
                                    <th>Fecha/Hora</th>
                                    <th>Tipo</th>
                                    <th>Motivo</th>
                                </tr>
                            </thead>
                            <tbody>
                                <?php
                                try {
                                    $stmt = $connect->prepare("
                                        SELECT a.*, p.modalidad, p.cump 
                                        FROM asistencia_qr a 
                                        LEFT JOIN patients p ON a.documento = p.numhs 
                                        ORDER BY a.fecha DESC 
                                        LIMIT 50
                                    ");
                                    $stmt->execute();
                                    $asistencias = $stmt->fetchAll();

                                    foreach ($asistencias as $row) {
                                        $tipo = isset($row['tipo']) ? $row['tipo'] : 'ENTRADA';
                                        $motivo = isset($row['motivo']) ? $row['motivo'] : '-';
                                        $modalidad = isset($row['modalidad']) ? $row['modalidad'] : 'N/A';
                                        
                                        // Calcular edad basada en fecha de nacimiento
                                        $edad = '';
                                        if(!empty($row['cump'])) {
                                            $fechaNacimiento = new DateTime($row['cump']);
                                            $hoy = new DateTime();
                                            $edad = $hoy->diff($fechaNacimiento)->y . ' años';
                                        } else {
                                            $edad = 'N/A';
                                        }
                                        
                                        $badgeClass = $tipo == 'ENTRADA' ? 'badge-success' : 'badge-warning';
                                        $tipoIcon = $tipo == 'ENTRADA' ? '🟢' : '🔴';
                                        
                                        echo "<tr>";
                                        echo "<td>{$row['id']}</td>";
                                        echo "<td>{$row['documento']}</td>";
                                        echo "<td>{$row['nompa']}</td>";
                                        echo "<td>{$row['apepa']}</td>";
                                        echo "<td>$edad</td>";
                                        echo "<td>".htmlspecialchars($modalidad)."</td>";
                                        echo "<td>".date('d/m/Y H:i', strtotime($row['fecha']))."</td>";
                                        echo "<td><span class='badge $badgeClass'>$tipoIcon $tipo</span></td>";
                                        echo "<td style='max-width: 200px; overflow: hidden; text-overflow: ellipsis; white-space: nowrap;' title='$motivo'>$motivo</td>";
                                        echo "</tr>";
                                    }

                                } catch (PDOException $e) {
                                    echo "<tr><td colspan='9'>Error: ".$e->getMessage()."</td></tr>";
                                }
                                ?>
                            </tbody>
                        </table>
                    </div>
                </div>
            </div>
        </main>
    </section>

    <script src="../../backend/js/jquery.min.js"></script>
    <script src="../../backend/js/script.js"></script>
    <script src="https://cdn.datatables.net/1.11.5/js/jquery.dataTables.min.js"></script>
    <script src="https://unpkg.com/html5-qrcode@2.3.8/html5-qrcode.min.js"></script>
    <script src="https://cdnjs.cloudflare.com/ajax/libs/jspdf/2.5.1/jspdf.umd.min.js"></script>
    <script src="https://cdnjs.cloudflare.com/ajax/libs/jspdf-autotable/3.5.31/jspdf.plugin.autotable.min.js"></script>

    <script>
        let dataTable;
        
        $(document).ready(function() {
            dataTable = $('#attendanceTable').DataTable({
                "language": {
                    "url": "//cdn.datatables.net/plug-ins/1.11.5/i18n/es-ES.json"
                },
                "order": [[0, "desc"]]
            });
            
            // Establecer fecha de hoy por defecto
            let today = new Date().toISOString().split('T')[0];
            document.getElementById('singleDate').value = today;
            document.getElementById('startDate').value = today;
            document.getElementById('endDate').value = today;
        });
        
        // Función para mostrar/ocultar campos de fecha
        function toggleDateInputs() {
            const filterType = document.getElementById('filterType').value;
            const singleDateGroup = document.getElementById('singleDateGroup');
            const startDateGroup = document.getElementById('startDateGroup');
            const endDateGroup = document.getElementById('endDateGroup');
            
            // Ocultar todos
            singleDateGroup.style.display = 'none';
            startDateGroup.style.display = 'none';
            endDateGroup.style.display = 'none';
            
            // Mostrar según selección
            if (filterType === 'single') {
                singleDateGroup.style.display = 'block';
            } else if (filterType === 'range') {
                startDateGroup.style.display = 'block';
                endDateGroup.style.display = 'block';
            }
        }
        
        // Función para aplicar filtro a la tabla
        function applyFilter() {
            const filterType = document.getElementById('filterType').value;
            const singleDate = document.getElementById('singleDate').value;
            const startDate = document.getElementById('startDate').value;
            const endDate = document.getElementById('endDate').value;
            const filterCentro = document.getElementById('filterCentro').value;
            
            let filterInfo = '';
            
            // Limpiar filtros anteriores
            $.fn.dataTable.ext.search = [];
            
            // Aplicar filtro combinado (fecha Y centro)
            $.fn.dataTable.ext.search.push(function(settings, data, dataIndex) {
                // Filtro de fecha (columna 5 - Fecha/Hora)
                let dateMatch = true;
                if (filterType === 'today') {
                    const today = new Date().toISOString().split('T')[0];
                    const rowDate = data[5].split(' ')[0].split('/').reverse().join('-');
                    dateMatch = rowDate === today;
                } else if (filterType === 'single' && singleDate) {
                    const rowDate = data[5].split(' ')[0].split('/').reverse().join('-');
                    dateMatch = rowDate === singleDate;
                } else if (filterType === 'range' && startDate && endDate) {
                    const rowDate = data[5].split(' ')[0].split('/').reverse().join('-');
                    dateMatch = rowDate >= startDate && rowDate <= endDate;
                }
                
                // Filtro de centro (columna 4 - Modalidad)
                let centroMatch = true;
                if (filterCentro !== 'all') {
                    const modalidadCell = data[4];
                    centroMatch = modalidadCell.includes(filterCentro);
                }
                
                return dateMatch && centroMatch;
            });
            
            // Construir texto de información del filtro
            if (filterType === 'today') {
                filterInfo = '(Hoy)';
            } else if (filterType === 'single' && singleDate) {
                filterInfo = '(' + singleDate + ')';
            } else if (filterType === 'range' && startDate && endDate) {
                filterInfo = '(' + startDate + ' a ' + endDate + ')';
            } else {
                filterInfo = '(Todas)';
            }
            
            if (filterCentro !== 'all') {
                filterInfo += ' - ' + filterCentro;
            }
            
            document.getElementById('filterInfo').textContent = filterInfo;
            dataTable.draw();
        }

        let html5QrcodeScanner;
        let currentCameraId = null;
        let cameras = [];
        
        function onScanSuccess(decodedText, decodedResult) {
            document.getElementById('detected-qr-code').value = decodedText;
            html5QrcodeScanner.stop().then(() => {
                document.querySelector(".detected-container").style.display = 'block';
            }).catch(err => {
                console.error("Error al detener scanner:", err);
                document.querySelector(".detected-container").style.display = 'block';
            });
        }
        
        function onScanError(errorMessage) {
            // Silenciar errores de escaneo (son normales cuando no hay QR visible)
        }
        
        function startScanner() {
            html5QrcodeScanner = new Html5Qrcode("scanner");
            
            // Obtener lista de cámaras
            Html5Qrcode.getCameras().then(devices => {
                cameras = devices;
                
                if (devices && devices.length > 0) {
                    // Preferir cámara trasera (environment)
                    let rearCamera = devices.find(device => 
                        device.label.toLowerCase().includes('back') ||
                        device.label.toLowerCase().includes('rear') ||
                        device.label.toLowerCase().includes('environment') ||
                        device.label.toLowerCase().includes('trasera')
                    );
                    
                    currentCameraId = rearCamera ? rearCamera.id : devices[0].id;
                    
                    // Configuración del scanner
                    const config = {
                        fps: 10,
                        qrbox: { width: 250, height: 250 },
                        aspectRatio: 1.0
                    };
                    
                    // Iniciar escaneo
                    html5QrcodeScanner.start(
                        currentCameraId,
                        config,
                        onScanSuccess,
                        onScanError
                    ).catch(err => {
                        console.error("Error al iniciar cámara:", err);
                        alert("Error al acceder a la cámara. Por favor, asegúrese de dar permisos de cámara al navegador.");
                    });
                    
                    // Mostrar/ocultar botón de cambio según cantidad de cámaras
                    if (cameras.length <= 1) {
                        document.querySelector('.switch-camera-btn').style.display = 'none';
                    }
                } else {
                    alert("No se encontraron cámaras en este dispositivo.");
                }
            }).catch(err => {
                console.error("Error al obtener cámaras:", err);
                alert("Error al acceder a las cámaras. Verifique los permisos del navegador.");
            });
        }
        
        function switchCamera() {
            if (cameras.length <= 1) return;
            
            // Encontrar índice actual
            let currentIndex = cameras.findIndex(cam => cam.id === currentCameraId);
            let nextIndex = (currentIndex + 1) % cameras.length;
            currentCameraId = cameras[nextIndex].id;
            
            // Detener y reiniciar con nueva cámara
            html5QrcodeScanner.stop().then(() => {
                const config = {
                    fps: 10,
                    qrbox: { width: 250, height: 250 },
                    aspectRatio: 1.0
                };
                
                html5QrcodeScanner.start(
                    currentCameraId,
                    config,
                    onScanSuccess,
                    onScanError
                ).catch(err => {
                    console.error("Error al cambiar cámara:", err);
                    alert("Error al cambiar de cámara");
                });
            }).catch(err => {
                console.error("Error al detener cámara:", err);
            });
        }
        
        function resetScanner() {
            document.querySelector(".detected-container").style.display = 'none';
            document.getElementById('formAsistencia').reset();
            document.getElementById('opcionesContainer').style.display = 'block';
            document.getElementById('eventualidadSelector').style.display = 'none';
            document.getElementById('tipoEventualidad').value = '';
            
            // Reiniciar scanner
            if (html5QrcodeScanner) {
                html5QrcodeScanner.stop().then(() => {
                    startScanner();
                }).catch(err => {
                    console.error("Error al resetear:", err);
                    startScanner();
                });
            } else {
                startScanner();
            }
        }
        
        // Función para registrar ENTRADA automáticamente
        function registrarEntrada() {
            document.getElementById('tipo_asistencia').value = 'ENTRADA';
            document.getElementById('modo_registro').value = 'NORMAL';
            document.getElementById('motivo_salida').value = '';
            document.getElementById('formAsistencia').submit();
        }
        
        // Función para registrar SALIDA automáticamente
        function registrarSalida() {
            document.getElementById('tipo_asistencia').value = 'SALIDA';
            document.getElementById('modo_registro').value = 'NORMAL';
            document.getElementById('motivo_salida').value = '';
            document.getElementById('formAsistencia').submit();
        }
        
        // Función para mostrar selector de eventualidad
        function mostrarEventualidad() {
            document.getElementById('opcionesContainer').style.display = 'none';
            document.getElementById('eventualidadSelector').style.display = 'block';
        }
        
        // Función para registrar EVENTUALIDAD automáticamente al seleccionar tipo
        function registrarEventualidad() {
            const tipoEventualidad = document.getElementById('tipoEventualidad').value;
            
            if (tipoEventualidad !== '') {
                document.getElementById('tipo_asistencia').value = 'SALIDA';
                document.getElementById('modo_registro').value = 'EVENTUALIDAD';
                document.getElementById('motivo_salida').value = tipoEventualidad;
                document.getElementById('formAsistencia').submit();
            }
        }
        
        document.addEventListener("DOMContentLoaded", startScanner);

        // Función para descargar datos de la tabla como CSV
        // Función mejorada para descargar datos de la tabla como CSV con columnas bien separadas
function downloadTableAsCSV(filename = 'asistencias.csv') {
    // Usaremos punto y coma como delimitador ya que es más compatible con Excel en español
    const delimiter = ';';
    let csv = [];
            // Encabezados
            let headers = [];
            document.querySelectorAll('#attendanceTable thead th').forEach(th => {
                headers.push(th.textContent.trim());
            });
            csv.push(headers.join(delimiter));
            // Filas de datos
            document.querySelectorAll('#attendanceTable tbody tr').forEach(tr => {
                let row = [];
                tr.querySelectorAll('td').forEach(td => {
                    // Extraer el texto y eliminar caracteres problemáticos
                    let text = td.textContent.trim();
                    // Eliminar emojis y limpiar
                    text = text.replace(/[🟢🔴]/g, '').trim();
                    // Eliminar cualquier delimitador del texto para evitar problemas
                    text = text.replace(new RegExp(delimiter, 'g'), ' ');
                    row.push(text);
                });
                csv.push(row.join(delimiter));
            });
    // Crear blob y descargar con BOM UTF-8 para mejor compatibilidad con Excel
    let csvContent = '\uFEFF' + csv.join('\r\n'); // BOM para UTF-8 y CRLF para compatibilidad
    let blob = new Blob([csvContent], { type: 'text/csv;charset=utf-8;' });
    let link = document.createElement('a');
    let url = URL.createObjectURL(blob);
    
    link.setAttribute('href', url);
    link.setAttribute('download', filename);
    link.style.visibility = 'hidden';
    
    document.body.appendChild(link);
    link.click();
    document.body.removeChild(link);
}

        // Agregar evento al botón de descarga CSV
        document.getElementById('downloadBtn').addEventListener('click', function() {
            let today = new Date();
            let date = today.getDate() + '-' + (today.getMonth() + 1) + '-' + today.getFullYear();
            downloadTableAsCSV('asistencias_' + date + '.csv');
        });

        // Función para descargar PDF con filtros
        document.getElementById('downloadPdfBtn').addEventListener('click', function() {
            const { jsPDF } = window.jspdf;
            const doc = new jsPDF();
            
            const filterType = document.getElementById('filterType').value;
            const singleDate = document.getElementById('singleDate').value;
            const startDate = document.getElementById('startDate').value;
            const endDate = document.getElementById('endDate').value;
            
            // Configuración del documento
            const pageWidth = doc.internal.pageSize.getWidth();
            const pageHeight = doc.internal.pageSize.getHeight();
            
            // Encabezado con gradiente simulado
            doc.setFillColor(102, 126, 234); // Color morado
            doc.rect(0, 0, pageWidth, 35, 'F');
            
            // Logo/Icono (círculo blanco)
            doc.setFillColor(255, 255, 255);
            doc.circle(20, 17, 8, 'F');
            
            // Título
            doc.setTextColor(255, 255, 255);
            doc.setFontSize(20);
            doc.setFont('helvetica', 'bold');
            doc.text('Programa Persona Mayor', 35, 15);
            
            doc.setFontSize(14);
            doc.setFont('helvetica', 'normal');
            let reportTitle = 'Reporte de Asistencias';
            
            // Agregar información del filtro al título
            if (filterType === 'today') {
                reportTitle += ' - Hoy';
            } else if (filterType === 'single' && singleDate) {
                reportTitle += ' - ' + singleDate;
            } else if (filterType === 'range' && startDate && endDate) {
                reportTitle += ' - ' + startDate + ' a ' + endDate;
            }
            
            doc.text(reportTitle, 35, 23);
            
            // Fecha del reporte
            let today = new Date();
            let dateStr = today.getDate() + '/' + (today.getMonth() + 1) + '/' + today.getFullYear();
            let timeStr = today.getHours() + ':' + String(today.getMinutes()).padStart(2, '0');
            doc.setFontSize(10);
            doc.text('Fecha: ' + dateStr + ' - Hora: ' + timeStr, 35, 30);
            
            // Estadísticas
            doc.setTextColor(0, 0, 0);
            doc.setFontSize(11);
            doc.setFont('helvetica', 'bold');
            doc.text('Resumen de Asistencias', 14, 45);
            
            doc.setFont('helvetica', 'normal');
            doc.setFontSize(10);
            
            <?php
            $total_pdf = $connect->query("SELECT COUNT(*) FROM asistencia_qr")->fetchColumn();
            $hoy_pdf = $connect->query("SELECT COUNT(*) FROM asistencia_qr WHERE DATE(fecha) = CURDATE()")->fetchColumn();
            ?>
            
            // Obtener datos del profesional
            const professionalName = document.getElementById('professionalName').value || 'No especificado';
            const professionalPosition = document.getElementById('professionalPosition').value || 'No especificado';
            
            // Preparar datos de la tabla PRIMERO (solo filas visibles después del filtro)
            let tableData = [];
            dataTable.rows({ search: 'applied' }).every(function() {
                let rowNode = this.node();
                let row = [];
                
                // Extraer datos de cada celda de la fila
                $(rowNode).find('td').each(function() {
                    let text = $(this).text().trim();
                    // Limpiar emojis
                    text = text.replace(/[🟢🔴🤒🚨]/g, '').trim();
                    row.push(text);
                });
                
                tableData.push(row);
            });
            
            // Verificar si hay datos
            if (tableData.length === 0) {
                alert('No hay datos para exportar con el filtro seleccionado');
                return;
            }
            
            // Mostrar estadísticas según el filtro
            let statsText = 'Registros en este reporte: ' + tableData.length;
            doc.text(statsText, 14, 52);
            
            if (filterType === 'single' && singleDate) {
                doc.text('Fecha: ' + singleDate, 14, 58);
            } else if (filterType === 'range' && startDate && endDate) {
                doc.text('Período: ' + startDate + ' al ' + endDate, 14, 58);
            } else if (filterType === 'today') {
                doc.text('Fecha: Hoy (' + dateStr + ')', 14, 58);
            } else {
                doc.text('Período: Todas las fechas', 14, 58);
            }
            
            // Crear tabla con autoTable
            doc.autoTable({
                startY: 65,
                head: [['#', 'Documento', 'Nombre', 'Apellido', 'Centro', 'Fecha/Hora', 'Tipo', 'Motivo']],
                body: tableData,
                theme: 'striped',
                headStyles: {
                    fillColor: [102, 126, 234],
                    textColor: [255, 255, 255],
                    fontStyle: 'bold',
                    fontSize: 8
                },
                bodyStyles: {
                    fontSize: 7,
                    textColor: [45, 52, 54]
                },
                alternateRowStyles: {
                    fillColor: [245, 246, 250]
                },
                margin: { top: 65, left: 8, right: 8 },
                columnStyles: {
                    0: { cellWidth: 8 },
                    1: { cellWidth: 20 },
                    2: { cellWidth: 22 },
                    3: { cellWidth: 22 },
                    4: { cellWidth: 30 },
                    5: { cellWidth: 28 },
                    6: { cellWidth: 18 },
                    7: { cellWidth: 30 }
                },
                didDrawPage: function(data) {
                    // Pie de página
                    doc.setFontSize(8);
                    doc.setTextColor(150);
                    doc.text(
                        'Página ' + doc.internal.getCurrentPageInfo().pageNumber,
                        pageWidth / 2,
                        pageHeight - 10,
                        { align: 'center' }
                    );
                }
            });
            
            // Agregar firma del profesional al final
            const finalY = doc.lastAutoTable.finalY || 65;
            
            // Espacio antes de la firma
            const signatureY = finalY + 30;
            
            // Línea de firma
            doc.setLineWidth(0.5);
            doc.line(14, signatureY, 90, signatureY);
            
            // Nombre del profesional
            doc.setFontSize(11);
            doc.setFont('helvetica', 'bold');
            doc.text(professionalName, 52, signatureY + 6, { align: 'center' });
            
            // Cargo del profesional
            doc.setFontSize(9);
            doc.setFont('helvetica', 'normal');
            doc.setTextColor(100);
            doc.text(professionalPosition, 52, signatureY + 12, { align: 'center' });
            
            // Descargar PDF
            doc.save('reporte_asistencias_' + dateStr.replace(/\//g, '-') + '.pdf');
        });
    </script>
</body>
</html>